"""
Django settings for base project.

Generated by 'django-admin startproject' using Django 3.2.12

For more information on this file, see
https://docs.djangoproject.com/en/3.2/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/3.2/ref/settings/
"""
import os
from pathlib import Path

# noinspection PyPackageRequirements
import environ
from corsheaders.defaults import default_headers

# Build paths inside the project like this: BASE_DIR / 'subdir'.
from firebase_admin import initialize_app
from django.utils.translation import gettext_lazy as _

BASE_DIR = Path(__file__).resolve().parent.parent.parent
PROJECT_NAME_SHORT = 'Local Secrets'
PROJECT_NAME = f'{PROJECT_NAME_SHORT} ' '({})'

CSRF_TRUSTED_ORIGINS = ["https://localsecrets.zoondia.org", "https://localsecrets-staging.rudo.es",]

# Read .env file
env = environ.Env(DEBUG=(bool, False))
# assert using format config.settings.{environment}
environment = env('DJANGO_SETTINGS_MODULE', default='')[16:]
if environment in ['local', 'test']:
    env.read_env(BASE_DIR.joinpath('.docker/environments/local.env'))

# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/3.2/howto/deployment/checklist/

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = env('DJANGO_SECRET_KEY')

# SECURITY WARNING: don't run with debug turned on in production!
DEBUG = env.bool('DJANGO_DEBUG', default=True)

ALLOWED_HOSTS = env.list('DJANGO_ALLOWED_HOSTS')

MODERN_ADMIN = env.bool('DJANGO_MODERN_ADMIN', default=False)


# Application definition
# https://docs.djangoproject.com/en/3.2/ref/settings/#std:setting-INSTALLED_APPS
DJANGO_CUSTOM_ADMIN = (
    [
        'jazzmin',
    ]
    if MODERN_ADMIN
    else []
)

DJANGO_APPS = DJANGO_CUSTOM_ADMIN + [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'django.contrib.postgres',
    'django.contrib.gis',
]

THIRD_PARTY_APPS = [
    'rest_framework',
    'corsheaders',
    'oauth2_provider',
    'easy_thumbnails',
    'social_django',
    'rest_framework_social_oauth2',
    'fcm_django',
    'dal',
    'dal_select2',
    'dal_admin_filters',
    'nested_admin',
    'import_export',
    'more_admin_filters',
    'drf_yasg',
    # 'storages',
]


LOCAL_APPS = [
    'local_secrets.app_version.apps.AppVersionConfig',
    'local_secrets.core.apps.CoreConfig',
    'local_secrets.messaging.apps.MessagingConfig',
    'local_secrets.twofa.apps.TwoFAConfig',
    'local_secrets.sites.apps.SitesConfig',
    'local_secrets.users.apps.UsersConfig',
    'local_secrets.cities.apps.CitiesConfig',
    'local_secrets.travels.apps.TravelsConfig',
    'local_secrets.routes.apps.RoutesConfig',
    'local_secrets.languages.apps.LanguagesConfig',
    'local_secrets.events.apps.EventsConfig',
    'local_secrets.searches.apps.SearchesConfig',
    'local_secrets.operations.apps.OperationsConfig',
    'local_secrets.service_integration.apps.ServiceIntegrationConfig',
]

INSTALLED_APPS = DJANGO_APPS + THIRD_PARTY_APPS + LOCAL_APPS

# https://docs.djangoproject.com/en/3.2/ref/middleware/
MIDDLEWARE = [
    'django.middleware.gzip.GZipMiddleware',
]

MIDDLEWARE += [
    'django.middleware.security.SecurityMiddleware',
    'corsheaders.middleware.CorsMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.locale.LocaleMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
    'local_secrets.core.middlewares.localization'
]

# Custom middlewares
MIDDLEWARE += [
    # 'local_secrets.core.middlewares.localization'
]


# https://docs.djangoproject.com/en/3.2/ref/settings/#root-urlconf
ROOT_URLCONF = 'config.urls'

# https://docs.djangoproject.com/en/3.2/ref/settings/#templates
TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        "DIRS": [BASE_DIR / "templates"],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
                'social_django.context_processors.backends',
                'social_django.context_processors.login_redirect',
            ],
        },
    },
]

WSGI_APPLICATION = 'config.wsgi.application'

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'level': 'DEBUG',
        },
        'file': {
            'class': 'logging.FileHandler',
            'level': 'DEBUG',
            'filename': os.path.join(BASE_DIR, 'logs', 'app.log'),
            'formatter': 'verbose',
        },
    },
    'formatters': {
        'verbose': {
            'format': '[{asctime}] {levelname} {name} - {message}',
            'style': '{',
        },
    },
    'loggers': {
        'django': {
            'level': 'ERROR',
            'propagate': True,
            'handlers': ['console', 'file'],
        },
        'local_secrets': {
            'level': 'INFO',
            'propagate': True,
            'handlers': ['console', 'file'],
        },
        '': {
            'level': 'ERROR',
            'propagate': True,
            'handlers': ['console', 'file'],
        },
    },
}

# Database
# https://docs.djangoproject.com/en/3.2/ref/settings/#databases
DEFAULT_AUTO_FIELD = 'django.db.models.BigAutoField'
DATABASES = {
    'default': {
        'ENGINE': 'django.contrib.gis.db.backends.postgis',
        'NAME': env('POSTGRES_DB'),
        'USER': env('POSTGRES_USER'),
        'PASSWORD': env('POSTGRES_PASSWORD'),
        'HOST': env('POSTGRES_HOST'),
        'PORT': env('POSTGRES_PORT'),
    }
}

print(DATABASES)


# Cache
# https://docs.djangoproject.com/en/3.2/ref/settings/#caches
CACHES = {
    'default': {
        'BACKEND': 'django.core.cache.backends.memcached.PyMemcacheCache',
        'LOCATION': env('CACHE_URL', default='127.0.0.1:11211'),
        'TIMEOUT': 600,
    }
}


# Internationalization
# https://docs.djangoproject.com/en/3.2/topics/i18n/
LANGUAGE_CODE = env('DJANGO_DEFAULT_LANGUAGE', default='en')

TIME_ZONE = env('DJANGO_TIME_ZONE', default='UTC')

USE_I18N = True

USE_L10N = True

USE_TZ = True

LOCALE_PATHS = [BASE_DIR.joinpath('locale')]

TIME_INPUT_FORMATS = ['%I:%M %p', '%H:%M']
TIME_FORMATS = ['%I:%M %p', '%H:%M']


LANGUAGES = [
    ('en', _('English')),
    ('es', _('Spanish')),
]


# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/3.2/howto/static-files/
STATIC_URL = '/static/'
STATIC_ROOT = 'static'

MEDIA_URL = '/media/'
MEDIA_ROOT = BASE_DIR.joinpath('media')

THUMBNAIL_ALIASES = {
    '': {
        'thumbnail': {'size': (128, 128), 'quality': 90},
        'midsize': {'size': (720, 720), 'quality': 90},
        'fullsize': {'size': (1080, 1080), 'quality': 90},
    },
}

WELL_KNOWN_URL = '/.well-known/'
WELL_KNOWN_ROOT = BASE_DIR.joinpath('.well-known')


# Email server
# https://docs.djangoproject.com/en/3.2/ref/settings/#email-backend
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = env('EMAIL_HOST')
EMAIL_PORT = env('EMAIL_PORT')
EMAIL_HOST_USER = env('EMAIL_HOST_USER')
EMAIL_HOST_PASSWORD = env('EMAIL_HOST_PASSWORD')
EMAIL_USE_TLS = True
DEFAULT_FROM_EMAIL = 'info@localsecrets.travel'

# Django Rest Framework
# https://www.django-rest-framework.org/api-guide/settings/
REST_FRAMEWORK = {
    'DEFAULT_AUTHENTICATION_CLASSES': (
        'rest_framework.authentication.SessionAuthentication',
        'oauth2_provider.contrib.rest_framework.OAuth2Authentication',
        'rest_framework_social_oauth2.authentication.SocialAuthentication',
    ),
    'COERCE_DECIMAL_TO_STRING': False,
    'EXCEPTION_HANDLER': 'local_secrets.core.exception_handlers.exception_handler',
    'DEFAULT_PARSER_CLASSES': (
        'rest_framework.parsers.JSONParser',
        'rest_framework.parsers.FormParser',
        'rest_framework.parsers.MultiPartParser',
    ),
    'DEFAULT_RENDERER_CLASSES': ('rest_framework.renderers.JSONRenderer',),
    'DEFAULT_PAGINATION_CLASS': 'rest_framework.pagination.PageNumberPagination',
    'PAGE_SIZE': 25,
}

OAUTH2_PROVIDER = {
    # One week (60 * 60 * 24 * 7 = 604800)
    'ACCESS_TOKEN_EXPIRE_SECONDS': 604800,
    'REFRESH_TOKEN_GRACE_PERIOD_SECONDS': 120,  # 2 minutes (60 * 2 = 120)
    'OAUTH_SINGLE_ACCESS_TOKEN': True,
    'OAUTH_DELETE_EXPIRED': True
}

AUTHENTICATION_BACKENDS = (
    'django.contrib.auth.backends.ModelBackend',
    'rest_framework_social_oauth2.backends.DjangoOAuth2',
    # Social auth available backends:
    # https://python-social-auth.readthedocs.io/en/latest/backends/
    # # Google OAuth2
    'social_core.backends.google.GoogleOAuth2',
    # # Facebook OAuth2
    'social_core.backends.facebook.FacebookAppOAuth2',
    'social_core.backends.facebook.FacebookOAuth2',
    # # Apple OAuth2
    'social_core.backends.apple.AppleIdAuth',
    # # Amazon OAuth2
    # 'social_core.backends.amazon.AmazonOAuth2',,
)


# OpenID Third-party configurations
# Facebook
SOCIAL_AUTH_FACEBOOK_KEY = env('FACEBOOK_ID')
SOCIAL_AUTH_FACEBOOK_SECRET = env('FACEBOOK_SECRET')
SOCIAL_AUTH_FACEBOOK_SCOPE = ['email']
SOCIAL_AUTH_FACEBOOK_PROFILE_EXTRA_PARAMS = {'fields': 'id, name, email'}

# Google
SOCIAL_AUTH_GOOGLE_OAUTH2_KEY = env('GOOGLE_AUTH_ID')
SOCIAL_AUTH_GOOGLE_OAUTH2_SECRET = env('GOOGLE_AUTH_SECRET')
SOCIAL_AUTH_GOOGLE_OAUTH2_SCOPE = [
    'https://www.googleapis.com/auth/userinfo.email',
    'https://www.googleapis.com/auth/userinfo.profile',
]

# Apple
#SOCIAL_AUTH_APPLE_ID_CLIENT = 'es.discover.localsecrets'
SOCIAL_AUTH_APPLE_ID_CLIENT = 'com.service.es.discover.localsecrets'
SOCIAL_AUTH_APPLE_ID_TEAM = env('SOCIAL_AUTH_APPLE_ID_TEAM')
SOCIAL_AUTH_APPLE_ID_KEY = env('SOCIAL_AUTH_APPLE_ID_KEY')
SOCIAL_AUTH_APPLE_ID_SECRET = env('SOCIAL_AUTH_APPLE_ID_SECRET')
SOCIAL_AUTH_APPLE_ID_SCOPE = ['email', 'name']
SOCIAL_AUTH_APPLE_ID_EMAIL_AS_USERNAME = True

# Amazon
SOCIAL_AUTH_AMAZON_KEY = '<your-key-id-goes-here>'
SOCIAL_AUTH_AMAZON_SECRET = '<your-app-secret-goes-here>'


# Sentry
if env.bool('SENTRY_ACTIVATED', default=False):
    import sentry_sdk
    from sentry_sdk.integrations.django import DjangoIntegration

    sentry_sdk.init(
        integrations=[DjangoIntegration()],
        # Set traces_sample_rate to 1.0 to capture 100% of transactions for performance monitoring. We recommend
        # adjusting this value in production.
        traces_sample_rate=0.01,
        # If you wish to associate users to errors (assuming you are using django.contrib.auth) you may enable
        # sending PII data.
        send_default_pii=True,
    )


# CORS config
# https://github.com/adamchainz/django-cors-headers#configuration
CORS_ALLOW_ALL_ORIGINS = True
CORS_ALLOW_CREDENTIALS = True
CORS_ALLOW_HEADERS = list(default_headers) + [
    'access-control-allow-origin',
    'language',
    'X-Platform',
    "ngrok-skip-browser-warning",
]

CORS_ALLOWED_ORIGINS = [
    "https://4b34-59-94-193-171.ngrok-free.app",
    "http://localhost:3000",
]

# remove
CORS_ORIGIN_WHITELIST = [
    "http://localhost:3000",
    "http://localhost:4230",
    "https://4b34-59-94-193-171.ngrok-free.app",
]


# Security configs
SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')
SECURE_SSL_REDIRECT = True
X_FRAME_OPTIONS = f'ALLOW-FROM {env("HTTP_HOST")}'


# Firebase
GOOGLE_APPLICATION_CREDENTIALS = BASE_DIR.joinpath(
    'local_secrets/google-services.json')
os.environ.setdefault('GOOGLE_APPLICATION_CREDENTIALS',
                      str(GOOGLE_APPLICATION_CREDENTIALS))
initialize_app()
FCM_DJANGO_SETTINGS = {
    "APP_VERBOSE_NAME": _('FCM'),
    "ONE_DEVICE_PER_USER": False,
    "DELETE_INACTIVE_DEVICES": False,
    "UPDATE_ON_DUPLICATE_REG_ID": True,
}


if DEBUG:
    import socket  # only if you haven't already imported this

    hostname, _, ips = socket.gethostbyname_ex(socket.gethostname())
    INTERNAL_IPS = [ip[: ip.rfind(".")] + ".1" for ip in ips] + [
        '127.0.0.1',
        '192.168.10.24',
    ]


# Jazzmin settings
JAZZMIN_SETTINGS = {
    'icons': {
        'app_version.AppVersion': 'fas fa-code-branch',
        'app_version.UserAppVersion': 'fas fa-user-cog',
        'app_version.Store': 'fas fa-store',
        'auth.user': 'fas fa-user',
        'auth.Group': 'fas fa-users',
        'oauth2_provider.Application': 'fas fa-rocket',
        'oauth2_provider.AccessToken': 'fas fa-key',
        'oauth2_provider.RefreshToken': 'fas fa-retweet',
        'quiz.Question': 'fas fa-question',
    },
}

AUTH_USER_MODEL = 'users.CustomUser'


IMAGE_MIN_SIZE = 5 * 1048576  # 5MB
IMAGE_MAX_SIZE = 12 * 1048576  # 12MB

DIRECTIONS_API_KEY = env('DIRECTIONS_API_KEY')

#
# # Spaces
# AWS_ACCESS_KEY_ID = 'DO00VFP9KVXW8QYP23FH'
# AWS_SECRET_ACCESS_KEY = 'kSfAYDtpVXuLVfIsphF1AfNYr8o/5qMuezHktIKP818'
#
# AWS_STORAGE_BUCKET_NAME = 'localsecrets-production'
# #AWS_S3_ENDPOINT_URL = 'https://localsecrets-production.fra1.digitaloceanspaces.com'
# AWS_S3_ENDPOINT_URL = 'https://localsecrets-production.fra1.digitaloceanspaces.com'
# AWS_S3_OBJECT_PARAMETERS = {
#     'CacheControl': 'max-age=86400',
# }
# AWS_DEFAULT_ACL = 'public-read'
#
# AWS_LOCATION = ''
#
# #AWS_S3_SIGNATURE_VERSION = 's3v4'
#
# STATICFILES_STORAGE = 'storages.backends.s3boto3.S3Boto3Storage'
# DEFAULT_FILE_STORAGE = 'local_secrets.core.storages.MediaStorage'
#
# #STATIC_URL = '{}/{}/'.format(AWS_S3_ENDPOINT_URL, AWS_LOCATION)
#
# #STATICFILES_DIRS = (os.path.join(BASE_DIR, 'statics'),)
#
# AWS_STATIC_LOCATION = 'static'
# #STATIC_ROOT = '/static/'
# STATIC_URL = '{}/{}/'.format(AWS_S3_ENDPOINT_URL, AWS_STATIC_LOCATION)
#
# AWS_MEDIA_LOCATION = 'media'
# MEDIA_ROOT = '{}/{}/'.format(AWS_S3_ENDPOINT_URL, AWS_MEDIA_LOCATION)
# MEDIA_URL = '/media/'
#
# THUMBNAIL_DEFAULT_STORAGE = 'local_secrets.core.storages.MediaStorage'


SIGATURE_SECRET = env("SIGATURE_SECRET")


# AWS configuartion
AWS_ACCESS_KEY_ID = env("AWS_ACCESS_KEY_ID")
AWS_SECRET_ACCESS_KEY = env("AWS_SECRET_ACCESS_KEY")
AWS_STORAGE_BUCKET_NAME = env("AWS_STORAGE_BUCKET_NAME")
AWS_S3_REGION_NAME = env("AWS_S3_REGION_NAME")

WEBSITE_URL = env("WEBSITE_URL")