
import json
import os
import random
import requests
import time
from datetime import datetime, timedelta
from django.core.management.base import BaseCommand
from django.utils.timezone import now, make_aware
from local_secrets.sites.models import Site, Comment
from django.contrib.auth import get_user_model

User = get_user_model()


class Command(BaseCommand):
    help = 'Import reviews from SerpAPI and fallback to local review generation'

    def add_arguments(self, parser):
        parser.add_argument('--ids-file', type=str, help="File with site IDs")
        parser.add_argument('--serpapi-key', type=str, required=True, help="Your SerpAPI key")
        parser.add_argument('--min-rating', type=float, default=4.5, help="Minimum SerpAPI rating required")
        parser.add_argument('--delay', type=float, default=0.5, help="Delay (seconds) between API calls")
        parser.add_argument('--reviews-file', type=str, default='reviews.json', help="Fallback local reviews file")

    def handle(self, *args, **options):
        ids_file = options.get('ids_file')
        serpapi_key = options['serpapi_key']
        min_rating = options.get('min_rating', 4.5)
        delay = options.get('delay', 0.5)
        reviews_file = options.get('reviews_file')

        # Initialize the script
        self.stdout.write(self.style.SUCCESS("Starting SerpAPI Reviews Import"))
        self.stdout.write(f"Parameters:\n- IDs file: {ids_file}\n- Min rating: {min_rating}\n- Delay: {delay}s\n- Reviews file: {reviews_file}\n")

        # Load and validate sites
        sites = self.load_sites(ids_file)
        if not sites.exists():
            self.stdout.write(self.style.ERROR("No sites found. Ensure site IDs exist or review the IDs file!"))
            return

        total_reviews_added = 0

        self.stdout.write(f"Processing {len(sites)} sites...")
        for site in sites:
            self.stdout.write(f"\nProcessing Site: {site.title} (ID: {site.id})")

            # Query SerpAPI reviews
            serpapi_reviews = self.fetch_reviews_from_serpapi(site, serpapi_key, min_rating)
            reviews_to_add = serpapi_reviews

            # Fallback to one hardcoded review if no SerpAPI reviews found
            if not reviews_to_add:
                self.stdout.write(f"→ No relevant SerpAPI reviews found. Falling back to one unique local review.")
                fallback_reviews = random.sample(self.generate_fallback_reviews(reviews_file), k=1)
                reviews_to_add = fallback_reviews

            # Add reviews to site
            reviews_added = self.add_reviews_to_site(site, reviews_to_add, delay)
            total_reviews_added += reviews_added
            self.stdout.write(self.style.SUCCESS(f"✓ Added {reviews_added} reviews to {site.title}\n"))

        self.stdout.write(self.style.SUCCESS(f"\n==== Import Complete: Total Reviews Added: {total_reviews_added} ===="))

    def load_sites(self, ids_file):
        """Load sites by ID from file or use all sites if no IDs file provided."""
        if ids_file and os.path.exists(ids_file):
            with open(ids_file, 'r') as f:
                site_ids = [line.strip() for line in f if line.strip()]
            return Site.objects.filter(id__in=site_ids)
        self.stdout.write(self.style.WARNING("No IDs file provided. Using all available sites."))
        return Site.objects.all()

    def fetch_reviews_from_serpapi(self, site, api_key, min_rating):
        """Query the SerpAPI for reviews."""
        url = f"https://serpapi.com/search.json?engine=google_maps_reviews&data_id={site.id}&api_key={api_key}"
        try:
            response = requests.get(url)
            if response.status_code == 200:
                reviews = response.json().get('reviews', [])
                return [
                    {
                        "review_text": r.get('snippet', 'Wonderful experience!'),
                        "rating": float(r.get('rating', 0)),
                        "created_at": self.generate_random_date()
                    }
                    for r in reviews if float(r.get('rating', 0)) >= min_rating
                ]
        except Exception as e:
            self.stdout.write(self.style.ERROR(f"Error querying SerpAPI for {site.id}: {str(e)}"))
        return []

    def generate_fallback_reviews(self, reviews_file):
        """Generate a fallback pool of ~100 diverse reviews."""
        reviews_pool = ["", "", "", "very,very good", "Very nice!", "Lovely!", "Outstanding service.", "Perfect spot.", "Love it.", "Great.", "Recommended.", "Good place.", "Friendly staff.", "A wonderful place.", "Revisit soon!", "Fantastic experience.", "Highly recommend.", "True local.", "Modern culture.", "Unique place.", "One-of-a-kind.", "Best hidden gem.", "Recommend for tourists.", "Local masterpiece.", "Great vibes.", "Natural beauty.", "Wonderful surroundings.", "Must-visit!", "Absolutely wonderful.", "Full of character.", "Perfect, peaceful.", "Charming, clean.", "Great to learn.", "Surpassed expectations.", "Peaceful retreat.", "Picturesque, worth it.", "Excellent value!", "Friendly locals.", "Delicious food.", "Ideal contemplation.", "Great location.", "Unique, breathtaking.", "Unparalleled experience.", "Sanctuary of charm.", "Best find!", "Joyful, colorful.", "Family fun!", "Serene destination.", "Lovely culture.", "Won't regret!", "Amazing escape.", "Very good.", "highly recommended.", "", "Cool!"]
 
        random.shuffle(reviews_pool)  # Shuffle to ensure randomness
        return [
            {
                "review_text": review,
                "rating": round(random.uniform(4.5, 5.0), 1),  # Random rating between 4.5 and 5.0
                "created_at": self.generate_random_date()
            }
            for review in reviews_pool
        ]

    def generate_random_date(self):
        """Generate a random date between Jan 1, 2024, and the current date."""
        start_date = datetime(2024, 1, 1)
        end_date = datetime(2025, 12, 31)
        random_date = start_date + timedelta(days=random.randint(0, (end_date - start_date).days))
        return make_aware(random_date)

    def add_reviews_to_site(self, site, reviews, delay):
        """Attach reviews to the given site."""
        user = User.objects.order_by("?").first()  # Select a random user
        count = 0
        for review in reviews:
            try:
                Comment.objects.create(
                    user=user,
                    site=site,
                    body=review['review_text'],
                    rating=review['rating'],
                    created_at=review['created_at']
                )
                self.stdout.write(self.style.SUCCESS(f"✓ Added review to {site.title}: {review['review_text'][:50]}..."))
                count += 1
                time.sleep(delay)
            except Exception as e:
                self.stdout.write(self.style.ERROR(f"Error adding review to site ({site.title}): {str(e)}"))
        return count



# To run the command, use:
# python manage.py serpapi_reviews --ids-file serpapi_reviews.txt --serpapi-key=fdcaa86dfef1b051789351905b1f99cf3c29743dd40ac1e42e73cc7e3961864e --min-rating=4.5 --delay=0.5 --reviews-file=basic_reviews.json