import { useState, useEffect } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
  Tabs,
  TabsContent,
  TabsList,
  TabsTrigger,
  Alert,
  AlertDescription,
  AlertTitle,
} from "@/components/ui";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';
import { ChartContainer, ChartTooltipContent } from "@/components/ui";

function SubscriptionDashboard() {
  const [statistics, setStatistics] = useState(null);
  const [billingHistory, setBillingHistory] = useState([]);
  const [usageHistory, setUsageHistory] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    // Fetch subscription data
    const fetchData = async () => {
      try {
        const [statsResponse, billingResponse, usageResponse] = await Promise.all([
          fetch('/api/subscription/summary'),
          fetch('/api/subscription/billing-history'),
          fetch('/api/subscription/usage-history')
        ]);

        const stats = await statsResponse.json();
        const billing = await billingResponse.json();
        const usage = await usageResponse.json();

        setStatistics(stats);
        setBillingHistory(billing);
        setUsageHistory(usage);
      } catch (error) {
        console.error('Error fetching data:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchData();
  }, []);

  if (isLoading) {
    return <div>Loading...</div>;
  }

  return (
    <div className="container mx-auto p-4 space-y-6">
      {/* Current Usage Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardHeader>
            <CardTitle>Sites Usage</CardTitle>
            <CardDescription>Current site allocation</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-4xl font-bold mb-2">
              {statistics?.sites.used}/{statistics?.sites.total}
            </div>
            <div className="text-muted-foreground">
              {statistics?.sites.percentage}% used
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Events Usage</CardTitle>
            <CardDescription>Current event allocation</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-4xl font-bold mb-2">
              {statistics?.events.used}/{statistics?.events.total}
            </div>
            <div className="text-muted-foreground">
              {statistics?.events.percentage}% used
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Subscription Status</CardTitle>
            <CardDescription>Current plan details</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-lg mb-2">
              Expires: {new Date(statistics?.subscription_ends).toLocaleDateString()}
            </div>
            <div className="text-muted-foreground">
              Auto-renew: {statistics?.auto_renew ? 'Enabled' : 'Disabled'}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Usage Alerts */}
      {(statistics?.sites.percentage > 80 || statistics?.events.percentage > 80) && (
        <Alert>
          <AlertTitle>High Usage Alert</AlertTitle>
          <AlertDescription>
            You are approaching your plan limits. Consider upgrading your subscription.
          </AlertDescription>
        </Alert>
      )}

      {/* Usage & Billing Tabs */}
      <Tabs defaultValue="usage" className="w-full">
        <TabsList>
          <TabsTrigger value="usage">Usage History</TabsTrigger>
          <TabsTrigger value="billing">Billing History</TabsTrigger>
        </TabsList>

        <TabsContent value="usage">
          <Card>
            <CardHeader>
              <CardTitle>Usage Trends</CardTitle>
              <CardDescription>Past 30 days usage statistics</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-[400px]">
                <ChartContainer>
                  <ResponsiveContainer width="100%" height="100%">
                    <LineChart data={usageHistory}>
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis 
                        dataKey="date" 
                        angle={-45} 
                        textAnchor="end"
                      />
                      <YAxis />
                      <Tooltip content={<ChartTooltipContent />} />
                      <Line 
                        type="monotone" 
                        dataKey="sites" 
                        name="Sites"
                        stroke="hsl(var(--chart-1))" 
                        fill="hsl(var(--chart-1))"
                      />
                      <Line 
                        type="monotone" 
                        dataKey="events" 
                        name="Events"
                        stroke="hsl(var(--chart-2))" 
                        fill="hsl(var(--chart-2))"
                      />
                    </LineChart>
                  </ResponsiveContainer>
                </ChartContainer>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="billing">
          <Card>
            <CardHeader>
              <CardTitle>Billing History</CardTitle>
              <CardDescription>Recent transactions</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="text-left py-2">Date</th>
                      <th className="text-left py-2">Description</th>
                      <th className="text-left py-2">Amount</th>
                      <th className="text-left py-2">Status</th>
                    </tr>
                  </thead>
                  <tbody>
                    {billingHistory.map((transaction, index) => (
                      <tr key={index} className="border-b">
                        <td className="py-2">
                          {new Date(transaction.date).toLocaleDateString()}
                        </td>
                        <td className="py-2">{transaction.description}</td>
                        <td className="py-2">${transaction.amount}</td>
                        <td className="py-2">
                          <span className={`inline-block px-2 py-1 rounded ${
                            transaction.status === 'completed' 
                              ? 'bg-green-100 text-green-800' 
                              : 'bg-yellow-100 text-yellow-800'
                          }`}>
                            {transaction.status}
                          </span>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}

export default SubscriptionDashboard;