# local_secrets/payment/config.py

from typing import Dict, Type
from .gateways.base import PaymentGatewayInterface
from .gateways.syspay import SysPayGateway
from .gateways.paypal import PayPalGateway
from .gateways.stripe import StripeGateway
from .gateways.sagepay import SagePayGateway 
from .gateways.mercado_pago import MercadoPagoGateway  
from .exceptions import PaymentError
from django.conf import settings

class PaymentConfig:
    """Configuration for payment gateways"""
    
    GATEWAY_MAPPING: Dict[str, Type[PaymentGatewayInterface]] = {
        'syspay': SysPayGateway,
        'paypal': PayPalGateway,
        'stripe': StripeGateway,
        'sagepay': SagePayGateway, 
        'mercado_pago': MercadoPagoGateway,  
    }
    
    @classmethod
    def get_gateway(cls, gateway_name: str) -> PaymentGatewayInterface:
        """Get gateway instance by name"""
        gateway_class = cls.GATEWAY_MAPPING.get(gateway_name)
        if not gateway_class:
            raise ValueError(f"Unknown gateway: {gateway_name}")
        return gateway_class()
