# local_secrets/payment/gateways/base.py

from abc import ABC, abstractmethod
from datetime import timezone
import logging
from typing import Dict, Any, Optional
from decimal import Decimal
from local_secrets.payment.models import VendorPayment

class PaymentGatewayInterface(ABC):
    """Abstract base class for payment gateway implementations"""
    
    @abstractmethod
    def process_payment(self, payment: VendorPayment) -> Dict[str, Any]:
        """
        Process a payment through the gateway
        Args:
            payment: VendorPayment instance to process
        Returns:
            Dict containing gateway response
        """
        pass
    
    @abstractmethod
    def refund_payment(self, payment: VendorPayment, amount: Decimal) -> Dict[str, Any]:
        """
        Process a refund through the gateway
        Args:
            payment: VendorPayment instance to refund
            amount: Amount to refund
        Returns:
            Dict containing refund response
        """
        pass
    
    @abstractmethod
    def verify_webhook(self, payload: Dict[str, Any], signature: str) -> bool:
        """
        Verify webhook signature
        Args:
            payload: Webhook payload
            signature: Signature to verify
        Returns:
            Boolean indicating if signature is valid
        """
        pass
    
    @abstractmethod
    def get_payment_status(self, payment_id: str) -> str:
        """
        Get payment status from gateway
        Args:
            payment_id: Payment ID to check
        Returns:
            String containing payment status
        """
        pass

