# local_secrets/payment/models.py

from django.db import models
from django.utils.translation import gettext_lazy as _
from django.utils import timezone
from django.core.exceptions import ValidationError
from django.core.validators import MaxValueValidator
from payments.models import BasePayment, PurchasedItem
from django.conf import settings

class VendorPayment(BasePayment):
    """Core payment model extending django-payments BasePayment"""
    
    vendor = models.ForeignKey(
        'users.Vendor',
        on_delete=models.CASCADE,
        related_name='vendor_payments'
    )
    subscription = models.ForeignKey(
        'users.VendorSubscription',
        on_delete=models.SET_NULL,
        null=True,
        related_name='vendor_payments'
    )
    
    # Additional fields not covered by BasePayment
    has_dispute = models.BooleanField(default=False)
    dispute_reason = models.CharField(max_length=255, blank=True)
    dispute_created_at = models.DateTimeField(null=True, blank=True)
    dispute_resolved_at = models.DateTimeField(null=True, blank=True)
    
    def get_failure_url(self):
        return f'/payments/{self.id}/failure/'
        
    def get_success_url(self):
        return f'/payments/{self.id}/success/'
        
    def get_purchased_items(self):
        yield PurchasedItem(
            name=self.subscription.plan.name if self.subscription else 'One-time payment',
            quantity=1,
            price=self.amount,
            currency=self.currency,
            sku=self.subscription.plan.id if self.subscription else None
        )

    def get_payment_method(self):
        """Get payment method for this payment"""
        return self.payment_method or self.subscription.vendor.default_payment_method

class Address(models.Model):
    """Address model for storing billing/shipping addresses"""
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    address_line1 = models.CharField(max_length=255)
    address_line2 = models.CharField(max_length=255, blank=True)
    city = models.CharField(max_length=100)
    state = models.CharField(max_length=100, blank=True)
    postal_code = models.CharField(max_length=20)
    country = models.CharField(max_length=3, default='US')
    phone_number = models.CharField(max_length=20, blank=True)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

class PaymentDispute(models.Model):
    """Dispute model for tracking payment disputes"""
    payment = models.ForeignKey(
        VendorPayment,
        on_delete=models.CASCADE,
        related_name='disputes'
    )
    reason = models.CharField(max_length=255)
    created_at = models.DateTimeField(auto_now_add=True)
    resolved_at = models.DateTimeField(null=True, blank=True)
    status = models.CharField(max_length=20, default='open')

    class Meta:
        verbose_name = _('Dispute')
        verbose_name_plural = _('Disputes')
        indexes = [
            models.Index(fields=['payment', 'status']),
            models.Index(fields=['created_at']),
        ]

class PaymentRefund(models.Model):
    """Refund model for tracking payment refunds"""
    payment = models.ForeignKey(
        VendorPayment,
        on_delete=models.CASCADE,
        related_name='refunds'
    )
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    reason = models.CharField(max_length=255, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        verbose_name = _('Refund')
        verbose_name_plural = _('Refunds')
        indexes = [
            models.Index(fields=['payment']),
            models.Index(fields=['created_at']),
        ]

class PaymentTransaction(models.Model):
    """Transaction model for tracking payment transactions"""
    payment = models.ForeignKey(
        VendorPayment,
        on_delete=models.CASCADE,
        related_name='transactions'
    )
    transaction_id = models.CharField(max_length=255, unique=True)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    status = models.CharField(max_length=20, default='pending')
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = _('Transaction')
        verbose_name_plural = _('Transactions')
        indexes = [
            models.Index(fields=['payment', 'status']),
            models.Index(fields=['created_at']),
        ]
                
class PaymentGateway(models.TextChoices):
    SYSPAY = 'syspay', _('SysPay')
    STRIPE = 'stripe', _('Stripe')
    PAYPAL = 'paypal', _('PayPal')
    RAZORPAY = 'razorpay', _('Razorpay')

class PaymentMethod(models.Model):
    """Enhanced PaymentMethod model with additional fields and methods"""
    vendor = models.ForeignKey('users.Vendor', on_delete=models.CASCADE, related_name='payment_methods', db_index=True)
    payment_type = models.CharField(max_length=20,
        choices=[
            ('stripe', 'Stripe'),
            ('paypal', 'PayPal'),
            ('syspay', 'SysPay'),
            ('razorpay', 'Razorpay')
        ]
    )
    is_default = models.BooleanField(default=False)
    created_at = models.DateTimeField(auto_now_add=True)
    
    gateway_payment_method_id = models.CharField(max_length=255)
    gateway_customer_id = models.CharField(max_length=255, blank=True)
    gateway_status = models.CharField(max_length=50, blank=True)

    # Gateway-specific fields
    stripe_payment_method_id = models.CharField(max_length=255, blank=True)
    paypal_payment_method_id = models.CharField(max_length=255, blank=True)
    razorpay_payment_method_id = models.CharField(max_length=255, blank=True)
    syspay_payment_method_id = models.CharField(max_length=255, blank=True)
    syspay_user_id = models.CharField(max_length=255, blank=True)
    syspay_status = models.CharField(
        max_length=50,
        choices=[
            ('pending', _('Pending')),
            ('active', _('Active')),
            ('suspended', _('Suspended')),
            ('terminated', _('Terminated'))
        ],
        blank=True
    )
    syspay_onboarding_complete = models.BooleanField(default=False)
    syspay_onboarding_url = models.URLField(blank=True)
    syspay_onboarding_error = models.CharField(max_length=255, blank=True)
    is_verified = models.BooleanField(default=False)
    verification_error = models.CharField(max_length=255, blank=True)

    class Meta:
        verbose_name = _('Payment Method')
        verbose_name_plural = _('Payment Methods')
        indexes = [
            models.Index(fields=['vendor', 'payment_type']),
            models.Index(fields=['created_at']),
        ]
        unique_together = ['vendor', 'gateway_payment_method_id']
        
    def save(self, *args, **kwargs):
        if self.is_default:
            PaymentMethod.objects.filter(
                vendor=self.vendor, 
                is_default=True
            ).exclude(id=self.id).update(is_default=False)
        super().save(*args, **kwargs)

    def get_payment_gateway(self):
        """Get the payment gateway for this method"""
        if self.payment_type == 'syspay':
            return PaymentGateway.SYSPAY
        elif self.payment_type == 'stripe':
            return PaymentGateway.STRIPE
        elif self.payment_type == 'paypal':
            return PaymentGateway.PAYPAL
        elif self.payment_type == 'razorpay':
            return PaymentGateway.RAZORPAY
        else:
            raise ValidationError(_('Unsupported payment type'))

class Payment(models.Model):
    """Enhanced Payment model with dispute handling and refund tracking"""
    PAYMENT_STATUS = [
        ('pending', _('Pending')),
        ('processing', _('Processing')),
        ('completed', _('Completed')),
        ('failed', _('Failed')),
        ('refunded', _('Refunded')),
        ('disputed', _('Disputed')),
        ('dispute_resolved', _('Dispute Resolved')),
        ('partially_refunded', _('Partially Refunded'))
    ]

    vendor = models.ForeignKey(
        'users.Vendor',
        on_delete=models.CASCADE,
        related_name='payments',
        db_index=True
    )
    subscription = models.ForeignKey(
        'users.VendorSubscription',
        on_delete=models.SET_NULL,
        null=True,
        related_name='payments'
    )
    payment_method = models.ForeignKey(
        PaymentMethod,
        on_delete=models.SET_NULL,
        null=True,
        related_name='payments'
    )
    
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    currency = models.CharField(max_length=3, default='USD')
    status = models.CharField(max_length=20, choices=PAYMENT_STATUS, default='pending')
    
    # Gateway reference and status
    payment_gateway_reference = models.CharField(max_length=255, blank=True, db_index=True)
    syspay_transaction_id = models.CharField(max_length=255, blank=True)
    syspay_payment_type = models.CharField(max_length=50, blank=True)
    syspay_payment_status = models.CharField(max_length=50, blank=True)
    
    # Refund tracking
    amount_refunded = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    refund_reason = models.CharField(max_length=255, blank=True)
    
    # Error handling
    error_message = models.TextField(blank=True)
    
    # Dispute handling
    has_dispute = models.BooleanField(default=False)
    dispute_reason = models.CharField(max_length=255, blank=True)
    dispute_created_at = models.DateTimeField(null=True, blank=True)
    dispute_resolved_at = models.DateTimeField(null=True, blank=True)
    
    # Timestamps
    created_at = models.DateTimeField(auto_now_add=True, db_index=True)
    updated_at = models.DateTimeField(auto_now=True)
    completed_at = models.DateTimeField(null=True, blank=True)

    class Meta:
        verbose_name = _('Payment')
        verbose_name_plural = _('Payments')
        indexes = [
            models.Index(fields=['vendor', 'status']),
            models.Index(fields=['created_at']),
            models.Index(fields=['payment_gateway_reference']),
        ]

    def mark_as_completed(self):
        """Mark payment as completed and update timestamps"""
        self.status = 'completed'
        self.completed_at = timezone.now()
        self.save()

    def process_refund(self, amount, reason):
        """Handle refund processing"""
        if amount > self.amount - self.amount_refunded:
            raise ValidationError(_('Refund amount exceeds available amount'))
        
        self.amount_refunded += amount
        self.refund_reason = reason
        
        if self.amount_refunded == self.amount:
            self.status = 'refunded'
        else:
            self.status = 'partially_refunded'
        
        self.save()

    def handle_dispute(self, reason):
        """Handle new dispute"""
        self.has_dispute = True
        self.dispute_reason = reason
        self.dispute_created_at = timezone.now()
        self.status = 'disputed'
        self.save()

    def resolve_dispute(self):
        """Handle dispute resolution"""
        self.has_dispute = False
        self.dispute_resolved_at = timezone.now()
        self.status = 'dispute_resolved'
        self.save()

    def process_syspay_payment(self, payment_data):
        """Process SysPay payment response"""
        self.syspay_transaction_id = payment_data.get('transaction_id')
        self.syspay_payment_type = payment_data.get('payment_type')
        self.syspay_payment_status = payment_data.get('status')
        
        if payment_data.get('status') == 'SUCCESS':
            self.mark_as_completed()
        elif payment_data.get('status') == 'FAILED':
            self.status = 'failed'
            self.error_message = payment_data.get('error_message')
        
        self.save()
      
class AutomaticBilling(models.Model):
    """Enhanced AutomaticBilling model with retry logic"""
    vendor = models.OneToOneField(
        'users.Vendor',
        on_delete=models.CASCADE,
        related_name='automatic_billing',
        db_index=True
    )
    payment_method = models.ForeignKey(
        PaymentMethod,
        on_delete=models.SET_NULL,
        null=True,
        related_name='automatic_billings'
    )
    enabled = models.BooleanField(default=False)
    billing_day = models.PositiveSmallIntegerField(
        validators=[MaxValueValidator(31)],
        help_text=_('Day of month for billing')
    )
    next_billing_date = models.DateField()
    
    gateway_subscription_id = models.CharField(max_length=255, blank=True)
    gateway_status = models.CharField(max_length=50, blank=True)
    gateway_error_message = models.TextField(blank=True)
    # SysPay specific fields
    syspay_subscription_id = models.CharField(max_length=255, blank=True)
    syspay_mandate_id = models.CharField(max_length=255, blank=True)
    
    # Retry logic
    retry_count = models.PositiveSmallIntegerField(default=0)
    last_retry_date = models.DateTimeField(null=True, blank=True)
    max_retries = models.PositiveSmallIntegerField(default=3)
    
    # Timestamps
    created_at = models.DateTimeField(auto_now_add=True, db_index=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = _('Automatic Billing')
        verbose_name_plural = _('Automatic Billings')
        indexes = [
            models.Index(fields=['vendor', 'enabled']),
            models.Index(fields=['next_billing_date']),
        ]

    def should_retry(self):
        """Check if payment should be retried"""
        return self.retry_count < self.max_retries

    def record_retry(self):
        """Record a retry attempt"""
        self.retry_count += 1
        self.last_retry_date = timezone.now()
        self.save()

    def reset_retries(self):
        """Reset retry counter after successful payment"""
        self.retry_count = 0
        self.last_retry_date = None
        self.save()

    def setup_syspay_recurring(self):
        """Setup recurring payments with SysPay"""
        from local_secrets.payment.providers import SysPayProvider
        
        provider = SysPayProvider()
        result = provider.setup_recurring_payment(
            user_id=self.payment_method.syspay_user_id,
            amount=self.vendor.subscription.plan.price,
            currency=self.vendor.subscription.plan.currency,
            billing_day=self.billing_day
        )
        
        self.syspay_subscription_id = result.get('subscription_id')
        self.syspay_mandate_id = result.get('mandate_id')
        self.save()

