
# local_secrets/payment/security/payment_security.py
from django.conf import settings
import logging

from local_secrets.payment.gateways import stripe


logger = logging.getLogger(__name__)


class PaymentSecurity:
    """Payment security handler"""
    
    @staticmethod
    def validate_webhook_signature(gateway_name: str, payload: bytes, signature: str) -> bool:
        """Validate webhook signature based on gateway"""
        validators = {
            'stripe': PaymentSecurity._validate_stripe_signature,
            'paypal': PaymentSecurity._validate_paypal_signature
        }
        
        validator = validators.get(gateway_name)
        if not validator:
            raise ValueError(f"Unsupported gateway: {gateway_name}")
            
        return validator(payload, signature)
        
    @staticmethod
    def _validate_stripe_signature(payload: bytes, signature: str) -> bool:
        """Validate Stripe webhook signature"""
        try:
            stripe.Webhook.construct_event(
                payload,
                signature,
                settings.STRIPE_WEBHOOK_SECRET
            )
            return True
        except Exception as e:
            logger.error(f"Stripe signature validation failed: {str(e)}")
            return False
