# local_secrets/payment/serializers.py 

from rest_framework import serializers
from .models import Payment, PaymentMethod, AutomaticBilling
from django.core.validators import MinValueValidator
from .models import  Address, VendorPayment, PaymentRefund, PaymentDispute, PaymentTransaction
from django.utils.translation import gettext_lazy as _
from django.core.exceptions import ValidationError

class VendorPaymentListSerializer(serializers.ModelSerializer):
    subscription_plan = serializers.CharField(
        source='subscription.plan.name',
        read_only=True
    )
    payment_method_details = serializers.SerializerMethodField()
    status_display = serializers.CharField(source='get_status_display')
    
    class Meta:
        model = VendorPayment
        fields = [
            'id',
            'created',
            'modified',
            'total',
            'currency',
            'status',
            'status_display',
            'payment_type',
            'transaction_id',
            'description',
            'billing_email',
            'subscription_plan',
            'payment_method_details',
            'error_message',
        ]

    def get_payment_method_details(self, obj):
        if obj.payment_method:
            return {
                'type': obj.payment_method.payment_type,
                'last4': obj.payment_method.last4,
                'brand': obj.payment_method.brand,
                'exp_month': obj.payment_method.exp_month,
                'exp_year': obj.payment_method.exp_year,
            }
        return None

class AddressSerializer(serializers.ModelSerializer):
    class Meta:
        model = Address
        fields = [
            'first_name',
            'last_name',
            'address_line1',
            'address_line2',
            'city',
            'state',
            'postal_code',
            'country',
            'phone_number'
        ]

class PaymentRefundSerializer(serializers.ModelSerializer):
    class Meta:
        model = PaymentRefund
        fields = [
            'id',
            'amount',
            'currency',
            'status',
            'created_at',
            'updated_at',
            'reason'
        ]

class PaymentDisputeSerializer(serializers.ModelSerializer):
    class Meta:
        model = PaymentDispute
        fields = [
            'id',
            'status',
            'created_at',
            'updated_at',
            'reason',
            'resolution'
        ]

class PaymentTransactionSerializer(serializers.ModelSerializer):
    class Meta:
        model = PaymentTransaction
        fields = [
            'id',
            'transaction_id',
            'amount',
            'currency',
            'status',
            'created_at',
            'updated_at',
            'description'
        ]
 
class VendorPaymentDetailSerializer(serializers.ModelSerializer):
    subscription_details = serializers.SerializerMethodField()
    payment_method_details = serializers.SerializerMethodField()
    billing_address = AddressSerializer(read_only=True)
    refunds = PaymentRefundSerializer(many=True, read_only=True)
    disputes = PaymentDisputeSerializer(many=True, read_only=True)
    transactions = PaymentTransactionSerializer(many=True, read_only=True)
    
    class Meta:
        model = VendorPayment
        fields = [
            'id',
            'created',
            'modified',
            'total',
            'currency',
            'status',
            'payment_type',
            'transaction_id',
            'description',
            'billing_email',
            'subscription_details',
            'payment_method_details',
            'billing_address',
            'refunds',
            'disputes',
            'transactions',
            'metadata',
            'error_message',
        ]

    def get_subscription_details(self, obj):
        if obj.subscription:
            return {
                'plan_name': obj.subscription.plan.name,
                'period_start': obj.subscription.current_period_start,
                'period_end': obj.subscription.current_period_end,
                'status': obj.subscription.status
            }
        return None

    def get_payment_method_details(self, obj):
        if obj.payment_method:
            return {
                'type': obj.payment_method.payment_type,
                'last4': obj.payment_method.last4,
                'brand': obj.payment_method.brand,
                'exp_month': obj.payment_method.exp_month,
                'exp_year': obj.payment_method.exp_year,
                'billing_details': obj.payment_method.billing_details
            }
        return None

class VendorPaymentSerializer(serializers.ModelSerializer):
    class Meta:
        model = VendorPayment
        fields = [
            'id', 'vendor', 'variant', 'status', 'total',
            'currency', 'created', 'modified', 'description'
        ]

class VendorPaymentCreateSerializer(serializers.ModelSerializer):
    class Meta:
        model = VendorPayment
        fields = ['variant', 'total', 'currency', 'description']

class AutomaticBillingSerializer(serializers.ModelSerializer):
    class Meta:
        model = AutomaticBilling
        fields = [
            'id', 'vendor', 'payment_reference', 'billing_day',
            'next_billing_date', 'enabled'
        ]

        read_only_fields = ['id', 'vendor', 'next_billing_date']
        
class SysPayTokenSerializer(serializers.Serializer):
    token = serializers.CharField(min_length=10, max_length=255)
    payment_type = serializers.CharField(
        default='syspay',
        validators=[lambda x: x == 'syspay' or ValueError("Invalid payment type")]
    )
    is_default = serializers.BooleanField(default=False)
    save_for_future = serializers.BooleanField(default=False)
    
    def validate_token(self, value):
        if not value.startswith('spt_'):
            raise serializers.ValidationError("Invalid SysPay token format")
        return value

class SysPayPaymentMethodSerializer(serializers.ModelSerializer):
    is_expired = serializers.BooleanField(read_only=True)
    last4 = serializers.CharField(read_only=True)
    card_type = serializers.CharField(read_only=True)
    expiry_date = serializers.DateField(read_only=True)
    
    class Meta:
        model = PaymentMethod
        fields = [
            'id', 
            'payment_type',
            'syspay_payment_method_id',
            'syspay_user_id',
            'syspay_status',
            'is_default',
            'is_verified',
            'is_expired',
            'last4',
            'card_type',
            'expiry_date',
            'created_at',
            'updated_at'
        ]
        read_only_fields = [
            'syspay_payment_method_id', 
            'syspay_user_id', 
            'syspay_status',
            'created_at',
            'updated_at'
        ]

class SysPayPaymentSerializer(serializers.ModelSerializer):
    payment_method = SysPayPaymentMethodSerializer(read_only=True)
    payment_method_id = serializers.IntegerField(write_only=True, required=False)
    amount = serializers.DecimalField(
        max_digits=10, 
        decimal_places=2,
        validators=[MinValueValidator(0.01)]
    )
    
    class Meta:
        model = Payment
        fields = [
            'id',
            'vendor',
            'amount',
            'currency',
            'status',
            'payment_method',
            'payment_method_id',
            'payment_gateway_reference',
            'created_at',
            'completed_at',
            'error_message',
            'metadata'
        ]
        read_only_fields = [
            'status', 
            'payment_gateway_reference', 
            'completed_at',
            'error_message'
        ]
    
    def validate(self, data):
        if not data.get('payment_method_id'):
            vendor = self.context['request'].user.vendor_profile
            default_payment_method = vendor.payment_methods.filter(
                is_default=True,
                is_active=True
            ).first()
            if not default_payment_method:
                raise serializers.ValidationError(
                    "No default payment method found. Please provide payment_method_id"
                )
            data['payment_method_id'] = default_payment_method.id
        return data

class SysPayWebhookSerializer(serializers.Serializer):
    event_type = serializers.CharField()
    transaction_id = serializers.CharField(required=False)
    payment_id = serializers.CharField(required=False)
    status = serializers.CharField()
    amount = serializers.DecimalField(
        max_digits=10, 
        decimal_places=2, 
        required=False
    )
    currency = serializers.CharField(required=False)
    error_code = serializers.CharField(required=False)
    error_message = serializers.CharField(required=False)
    metadata = serializers.DictField(required=False)
    
    def validate_event_type(self, value):
        valid_events = [
            'payment.success',
            'payment.failed',
            'payment.refunded',
            'payment.disputed',
            'subscription.created',
            'subscription.updated',
            'subscription.cancelled'
        ]
        if value not in valid_events:
            raise serializers.ValidationError(f"Invalid event type: {value}")
        return value

class AutomaticBillingSysPaySerializer(serializers.ModelSerializer):
    next_retry_date = serializers.DateTimeField(read_only=True)
    
    class Meta:
        model = AutomaticBilling
        fields = [
            'id',
            'enabled',
            'billing_day',
            'next_billing_date',
            'payment_method',
            'syspay_subscription_id',
            'retry_count',
            'next_retry_date',
            'last_payment_date',
            'last_payment_status'
        ]
        read_only_fields = [
            'syspay_subscription_id', 
            'retry_count',
            'next_retry_date',
            'last_payment_date',
            'last_payment_status'
        ]
        extra_kwargs = {
            'payment_method': {'required': True},
            'billing_day': {
                'min_value': 1,
                'max_value': 28
            }
        }
    
    def validate_payment_method(self, value):
        if not value.is_active or not value.is_verified:
            raise serializers.ValidationError(
                "Payment method must be active and verified"
            )
        return value