# local_secrets/payment/tests/test_syspay.py

import pytest
from django.test import TestCase
from django.contrib.auth import get_user_model
from unittest.mock import patch, MagicMock
from django.utils.text import slugify
from local_secrets.payment.providers.syspay_provider import SysPayProvider
from local_secrets.payment.models import PaymentMethod, Payment
from local_secrets.users.models import Vendor, BusinessType, BusinessCategory
from decimal import Decimal
  
 # local_secrets/payment/tests/test_syspay.py

import pytest
from django.test import TransactionTestCase
from django.contrib.auth import get_user_model
from django.db import connection
from unittest.mock import patch, MagicMock
from decimal import Decimal
from django.utils.text import slugify

from local_secrets.users.models import (
    Vendor, 
    BusinessType, 
    BusinessCategory
)
from local_secrets.payment.models import PaymentMethod, Payment
from local_secrets.payment.providers.syspay_provider import SysPayProvider

User = get_user_model()

class TestSysPayBase(TransactionTestCase):
    """Base test class with database configuration"""
    
    databases = {'default'}

    @classmethod
    def setUpClass(cls):
        """Set up test database"""
        super().setUpClass()
        
        # Create PostGIS extensions if needed
        with connection.cursor() as cursor:
            cursor.execute("CREATE EXTENSION IF NOT EXISTS postgis;")
            cursor.execute("CREATE EXTENSION IF NOT EXISTS postgis_topology;")

    def setUp(self):
        """Set up test data"""
        # Reset sequences
        with connection.cursor() as cursor:
            cursor.execute("""
                DO $$
                BEGIN
                    IF EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'users_vendor') THEN
                        PERFORM setval(pg_get_serial_sequence('users_vendor', 'id'), 1, false);
                    END IF;
                    IF EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'users_businesstype') THEN
                        PERFORM setval(pg_get_serial_sequence('users_businesstype', 'id'), 1, false);
                    END IF;
                    IF EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'users_businesscategory') THEN
                        PERFORM setval(pg_get_serial_sequence('users_businesscategory', 'id'), 1, false);
                    END IF;
                END $$;
            """)

        # Create test user with username
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123',
            first_name='Test',
            last_name='User'
        )

        # Create business category first
        self.business_category = BusinessCategory.objects.create(
            name='Test Category',
            slug=slugify('Test Category'),
            description='Test Category Description',
            icon='test-icon'
        )

        # Create test business type with category
        self.business_type = BusinessType.objects.create(
            category=self.business_category,
            name='Test Business',
            slug=slugify('Test Business'),
            description='Test Description',
            service_level='standard',
            target_audience='families',
            operating_season='year_round'
        )

        # Create test vendor
        self.vendor = Vendor.objects.create(
            user=self.user,
            company_name='Test Company',
            business_type=self.business_type,
            contact_email='vendor@example.com',
            slug=slugify('Test Company'),
            tax_id='TEST123',
            website='https://testcompany.com',
            description='Test vendor description',
            short_description='Short description',
            verification_status='pending'
        )

        # Initialize SysPay provider
        self.syspay_provider = SysPayProvider()

    def tearDown(self):
        """Clean up after tests"""
        Payment.objects.all().delete()
        PaymentMethod.objects.all().delete()
        Vendor.objects.all().delete()
        User.objects.all().delete()
        BusinessType.objects.all().delete()
        BusinessCategory.objects.all().delete()
        super().tearDown()

class TestSysPayIntegration(TestSysPayBase):
    """Test SysPay integration functionality"""

    def _create_test_user(self, suffix=""):
        """Helper method to create test user"""
        return User.objects.create_user(
            username=f'testuser{suffix}',
            email=f'test{suffix}@example.com',
            password='testpass123',
            first_name=f'Test{suffix}',
            last_name='User'
        )

    def _create_test_business_category(self, suffix=""):
        """Helper method to create test business category"""
        return BusinessCategory.objects.create(
            name=f'Test Category{suffix}',
            slug=slugify(f'Test Category{suffix}'),
            description=f'Test Category Description{suffix}',
            icon=f'test-icon{suffix}'
        )

    def _create_test_business_type(self, category, suffix=""):
        """Helper method to create test business type"""
        return BusinessType.objects.create(
            category=category,
            name=f'Test Business{suffix}',
            slug=slugify(f'Test Business{suffix}'),
            description=f'Test Description{suffix}',
            service_level='standard',
            target_audience='families',
            operating_season='year_round'
        )

    def _create_test_vendor(self, user, business_type, suffix=""):
        """Helper method to create test vendor"""
        return Vendor.objects.create(
            user=user,
            company_name=f'Test Company{suffix}',
            business_type=business_type,
            contact_email=f'vendor{suffix}@example.com',
            slug=slugify(f'Test Company{suffix}'),
            tax_id=f'TEST123{suffix}',
            website=f'https://testcompany{suffix}.com',
            description=f'Test vendor description{suffix}',
            short_description=f'Short description{suffix}',
            verification_status='pending'
        )

    @patch('requests.Session.request')
    def test_create_payment_method(self, mock_request):
        """Test creating a payment method through SysPay"""
        # Create test data
        test_user = self._create_test_user("_payment")
        test_category = self._create_test_business_category("_payment")
        test_business_type = self._create_test_business_type(test_category, "_payment")
        test_vendor = self._create_test_vendor(test_user, test_business_type, "_payment")

        # Mock response
        mock_response = MagicMock()
        mock_response.json.return_value = {
            'id': 'pm_test123',
            'status': 'success',
            'type': 'card'
        }
        mock_response.status_code = 200
        mock_request.return_value = mock_response

        # Test data
        payment_data = {
            'type': 'card',
            'currency': 'USD'
        }

        # Create payment method
        result = self.syspay_provider.setup_payment_method(
            vendor=test_vendor,
            payment_data=payment_data
        )

        # Assertions
        self.assertEqual(result['payment_method_id'], 'pm_test123')
        self.assertEqual(mock_request.call_count, 1)

    @patch('requests.Session.request')
    def test_process_payment(self, mock_request):
        """Test processing a payment through SysPay"""
        # Create test data
        test_user = self._create_test_user("_process")
        test_category = self._create_test_business_category("_process")
        test_business_type = self._create_test_business_type(test_category, "_process")
        test_vendor = self._create_test_vendor(test_user, test_business_type, "_process")

        # Create payment method
        payment_method = PaymentMethod.objects.create(
            vendor=test_vendor,
            syspay_payment_method_id='pm_test123',
            payment_type='card',
            is_default=True
        )

        # Mock response
        mock_response = MagicMock()
        mock_response.json.return_value = {
            'id': 'pay_test123',
            'status': 'success',
            'amount': '100.00',
            'currency': 'USD'
        }
        mock_response.status_code = 200
        mock_request.return_value = mock_response

        # Process payment
        payment_data = {
            'payment_method_id': payment_method.syspay_payment_method_id,
            'amount': Decimal('100.00'),
            'currency': 'USD'
        }

        result = self.syspay_provider.process_server_payment(payment_data)

        # Assertions
        self.assertEqual(result['id'], 'pay_test123')
        self.assertEqual(result['status'], 'success')

    def test_webhook_handling(self):
        """Test webhook handling"""
        webhook_payload = {
            'type': 'payment.succeeded',
            'data': {
                'object': {
                    'id': 'pay_test123',
                    'status': 'succeeded',
                    'amount': '100.00'
                }
            }
        }

        webhook_headers = {
            'X-Signature': 'test_signature',
            'X-Timestamp': '2025-04-08 10:00:00'
        }

        with patch.object(self.syspay_provider, 'verify_webhook', return_value=True):
            result = self.syspay_provider.handle_webhook_event(
                webhook_payload,
                webhook_headers
            )

        self.assertEqual(result['status'], 'processed')
        self.assertEqual(result['payment_id'], 'pay_test123')
