# local_secrets/payment/utility.py

from django.utils import timezone
from django.conf import settings
from django.core.cache import cache
from django.db import transaction
from django.db.models import Q
from django.urls import reverse
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.utils.html import strip_tags
from django.utils.translation import gettext as _
from django.utils.translation import gettext_lazy as _
from typing import Dict, Any, Optional
from decimal import Decimal
from pydantic import ValidationError

from local_secrets.sites.services.site_visibility import SiteVisibilityManager
from local_secrets.users.models import Vendor 
from local_secrets.payment.exceptions import PaymentError  
from local_secrets.payment.models import VendorPayment, PaymentMethod
from local_secrets.payment.exceptions import PaymentError


def get_payment_url(payment: VendorPayment) -> str:
    """Generate a payment URL for the given payment"""
    return reverse('payment:process', args=[payment.id])
 
def get_payment_status_url(payment: VendorPayment) -> str:
    """Generate a payment status URL for the given payment"""
    return reverse('payment:status', args=[payment.id])

def send_payment_notification(payment: VendorPayment, success: bool) -> None:
    """Send payment notification to the user"""
    subject = _("Payment Successful") if success else _("Payment Failed")
    message = render_to_string('payment/payment_notification.html', {'payment': payment, 'success': success})
    plain_message = strip_tags(message)
    send_mail(subject, plain_message, settings.DEFAULT_FROM_EMAIL, [payment.vendor.email], html_message=message)

def clear_payment_cache(payment_id: str) -> None:
    """Clear payment cache for the given payment ID"""
    cache_key = f'payment_{payment_id}'
    cache.delete(cache_key)
    cache.delete(f'payment_status_{payment_id}')

def get_payment_cache(payment_id: str) -> Optional[VendorPayment]:
    """Get payment from cache if available"""
    cache_key = f'payment_{payment_id}'
    payment = cache.get(cache_key)
    if not payment:
        try:
            payment = VendorPayment.objects.get(id=payment_id)
            cache.set(cache_key, payment, timeout=settings.CACHE_TIMEOUT)
        except VendorPayment.DoesNotExist:
            return None
    return payment
 
def validate_payment_method(payment_method: PaymentMethod) -> bool:
    """Validate the payment method"""
    if not payment_method.is_valid():
        raise ValidationError(_("Invalid payment method"))
    return True
 
def process_payment(payment: VendorPayment) -> None:
    """Process the payment"""
    try:
        with transaction.atomic():
            from local_secrets.payment.services import PaymentService 
            payment_service = PaymentService(payment.gateway)
            payment_service.create_payment(payment.vendor, payment.amount, payment.currency, payment.payment_method)
            payment.status = 'processed' 
            payment.save()
            send_payment_notification(payment, success=True)
            SiteVisibilityManager.handle_payment_status_change(payment)
    except Exception as e:
        payment.status = 'failed'
        payment.save()
        send_payment_notification(payment, success=False)
        raise PaymentError(_("Payment processing failed: %s") % str(e))

def refund_payment(payment: VendorPayment, amount: Decimal) -> None:
    """Process a refund for the given payment"""
    try:
        with transaction.atomic():
            from local_secrets.payment.services import PaymentService
            payment_service = PaymentService(payment.gateway)
            payment_service.refund_payment(payment, amount)
            payment.status = 'refunded'
            payment.save()
    except Exception as e:
        raise PaymentError(_("Refund processing failed: %s") % str(e))
 
def get_payment_status(payment: VendorPayment) -> str:
    """Get the payment status"""
    try:
        from local_secrets.payment.services import PaymentService
        payment_service = PaymentService(payment.gateway)
        status = payment_service.get_payment_status(payment.id)
        return status
    except Exception as e:
        raise PaymentError(_("Failed to retrieve payment status: %s") % str(e))
