# local_secrets/payment/webhooks.py

from typing import Dict, Any, Optional
from django.http import HttpResponse, JsonResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST
from django.conf import settings
from django.core.exceptions import ValidationError
from django.views import View
from django.db import transaction
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status
import json
import logging

from local_secrets.payment.gateways.paypal import PayPalGateway 
from local_secrets.payment.gateways.stripe import StripeGateway
from local_secrets.sites.models import Site, SiteVisibilityChoices
from .services import PaymentOrchestrator, PaymentService, RefundService, DisputeService 
from .exceptions import PaymentError
from .models import PaymentTransaction, VendorPayment 

logger = logging.getLogger(__name__)

class WebhookHandler:
    """Base webhook handler class"""
    
    def __init__(self, gateway_name: str):
        self.gateway_name = gateway_name
        self.gateway = self._get_gateway()
        self.payment_service = PaymentService(self.gateway)
        self.refund_service = RefundService(self.gateway)
        self.dispute_service = DisputeService(self.gateway)

    def _get_gateway(self):
        """Get payment gateway instance"""
        gateways = {
            'stripe': StripeGateway,
            'paypal': PayPalGateway, 
        }
        
        gateway_class = gateways.get(self.gateway_name)
        if not gateway_class:
            raise PaymentError(f"Unsupported payment gateway: {self.gateway_name}")
            
        return gateway_class()

    def _verify_signature(self, payload: bytes, signature: str) -> bool:
        """Verify webhook signature"""
        try:
            return self.gateway.verify_webhook(payload, signature)
        except Exception as e:
            logger.error(f"Signature verification failed: {str(e)}")
            return False

    def _get_signature_header(self, headers: Dict) -> Optional[str]:
        """Get signature from headers based on gateway"""
        signature_headers = {
            'stripe': 'Stripe-Signature',
            'paypal': 'Paypal-Transmission-Sig',
            'sofort': 'Sofort-Signature'
        }
        return headers.get(signature_headers.get(self.gateway_name))

    def process_webhook(self, request) -> Dict[str, Any]:
        """Process webhook request"""
        try:
            signature = self._get_signature_header(request.headers)
            if not signature:
                raise PaymentError("Missing webhook signature")

            if not self._verify_signature(request.body, signature):
                raise PaymentError("Invalid webhook signature")

            payload = json.loads(request.body)
            logger.info(f"Received {self.gateway_name} webhook: {payload.get('type')}")
            
            return self._process_event(payload)
            
        except json.JSONDecodeError as e:
            raise PaymentError(f"Invalid JSON payload: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Webhook processing failed: {str(e)}")

    def _process_event(self, payload: Dict[str, Any]) -> Dict[str, Any]:
        """Process webhook event based on type"""
        event_type = payload.get('type')
        
        handlers = {
            # Payment events
            'payment_intent.succeeded': self._handle_payment_success,
            'payment_intent.payment_failed': self._handle_payment_failure,
            'charge.succeeded': self._handle_charge_success,
            'charge.failed': self._handle_charge_failure,
            
            # Refund events
            'charge.refunded': self._handle_refund,
            'refund.succeeded': self._handle_refund_success,
            'refund.failed': self._handle_refund_failure,
            
            # Dispute events
            'charge.dispute.created': self._handle_dispute_created,
            'charge.dispute.updated': self._handle_dispute_updated,
            'charge.dispute.closed': self._handle_dispute_closed,
            
            # PayPal specific events
            'PAYMENT.CAPTURE.COMPLETED': self._handle_paypal_payment_success,
            'PAYMENT.CAPTURE.DENIED': self._handle_paypal_payment_failure,
        }
        
        handler = handlers.get(event_type)
        if handler:
            return handler(payload)
        
        logger.warning(f"Unhandled webhook event type: {event_type}")
        return {'status': 'ignored', 'message': f'Unhandled event type: {event_type}'}
 
    @transaction.atomic
    def _handle_payment_success(self, payload: Dict[str, Any]) -> Dict[str, Any]:
        """Handle successful payment"""
        try:
            payment_intent = payload['data']['object']
            payment = VendorPayment.objects.get(
                payment_gateway_reference=payment_intent['id']
            )
            
            orchestrator = PaymentOrchestrator(payment)
            result = orchestrator.handle_successful_payment(payment_intent)
            
            logger.info(
                "Payment processed successfully",
                extra={'payment_id': payment.id, **result}
            )
            
            return result
            
        except Exception as e:
            logger.error(f"Error handling payment success: {str(e)}")
            raise

    @transaction.atomic
    def _handle_payment_failure(self, payload: Dict[str, Any]) -> Dict[str, Any]:
        """Handle failed payment"""
        try:
            payment_intent = payload['data']['object']
            payment = VendorPayment.objects.get(
                payment_gateway_reference=payment_intent['id']
            )
            
            orchestrator = PaymentOrchestrator(payment)
            result = orchestrator.handle_failed_payment(payment_intent)
            
            logger.info(
                "Payment failure processed",
                extra={'payment_id': payment.id, **result}
            )
            
            return result
            
        except Exception as e:
            logger.error(f"Error handling payment failure: {str(e)}")
            raise
