# local_secrets/routes/admin.py

from django.contrib import admin
from django.contrib.admin import TabularInline
from import_export import fields
from import_export.admin import ImportExportModelAdmin
from import_export.resources import ModelResource
from import_export.widgets import ManyToManyWidget
from more_admin_filters import RelatedDropdownFilter, MultiSelectRelatedDropdownFilter
from django.utils.translation import gettext as _

from .models import Route, RouteStop, TranslatedRoute
from ..core.admin import admin_site
from ..sites.models import Site
from ..users.models import Ambassador, Tag


class StopInLine(admin.TabularInline):
    autocomplete_fields = [
        'site',
    ]
    model = RouteStop
    extra = 0
    verbose_name = _("Route Stop")
    verbose_name_plural = _("Route Stops")
    fields = ('site', 'order')
    ordering = ('order',)


class RouteResource(ModelResource):
    tags = fields.Field(column_name='tags', attribute='tags', widget=ManyToManyWidget(Tag))
    stops = fields.Field(column_name='stops', attribute='stops', widget=ManyToManyWidget(Site))
    cities = fields.Field(column_name='cities', attribute='cities', widget=ManyToManyWidget(Site))

    class Meta:
        model = Route
        fields = (
            'id',
            'title',
            'cities',
            'tags',
            'stops',
            'num_of_views',
            'is_top_ten',
            'sort_order',
        )
        import_id_fields = ('id',)
        export_order = fields


class TranslatedRouteInLine(TabularInline):
    model = TranslatedRoute
    extra = 0
    verbose_name = _("Translation")
    verbose_name_plural = _("Translations")
    fields = ('language', 'title')


class RouteAdmin(ImportExportModelAdmin, admin.ModelAdmin):
    resource_class = RouteResource

    def get_queryset(self, request):
        routes = Route.objects.all()
        try:
            ambassador = Ambassador.objects.get(id=request.user.id)
            routes = routes.filter(cities__id__in=ambassador.cities.values_list('id', flat=True))
        except Ambassador.DoesNotExist:
            pass
        return routes

    list_display = ('title', 'num_of_views', 'is_top_ten', 'sort_order', 'get_cities', 'get_tags')
    list_editable = ('is_top_ten', 'sort_order')
    filter_horizontal = ('tags', 'cities')
    autocomplete_fields = ('tags', 'cities')
    search_fields = ('title', 'cities__name', 'tags__title')
    list_filter = (
        ('cities', MultiSelectRelatedDropdownFilter),
        ('tags', MultiSelectRelatedDropdownFilter),
        'is_top_ten',
    )
    inlines = [TranslatedRouteInLine, StopInLine]
    
    fieldsets = (
        (None, {
            'fields': ('title', 'cities', 'tags', 'is_top_ten', 'sort_order')
        }),
        ('Statistics', {
            'fields': ('num_of_views',),
            'classes': ('collapse',)
        }),
    )
    
    def get_cities(self, obj):
        return ", ".join([city.name for city in obj.cities.all()[:3]])
    get_cities.short_description = 'Cities'
    
    def get_tags(self, obj):
        return ", ".join([tag.title for tag in obj.tags.all()[:3]])
    get_tags.short_description = 'Tags'


admin_site.register(Route, RouteAdmin)
