# local_secrets/routes/serializers.py

from django.core.cache import cache
from rest_framework import serializers

from local_secrets.cities.models import City
from local_secrets.cities.serializers import AddressSerializer, BaseCitySerializer, CitySerializer
from local_secrets.core.serializers import ThumbnailJSONSerializer
from local_secrets.routes.models import Route, RouteStop
from local_secrets.sites.models import Site
from local_secrets.sites.serializers import (
    CategoryListSerializer,
    LevelListSerializer,
    SiteListSerializer,
    SiteWithoutCitySerializer,
    SubCategorySerializer,
)
from local_secrets.users.serializers import TagOutputWithoutSelectionSerializer


class BaseRouteSerializer(serializers.ModelSerializer):
    """Base Route serializer with common fields and methods."""
    tags = serializers.SerializerMethodField()
    image = serializers.SerializerMethodField()
    sort_order = serializers.IntegerField(required=False)

    def get_tags(self, obj):
        cache_key = f'route_tags_{obj.id}'
        cached_data = cache.get(cache_key)
        if cached_data:
            return cached_data

        data = TagOutputWithoutSelectionSerializer(
            obj.tags.only('id', 'title').all(), 
            many=True, 
            read_only=True
        ).data
        cache.set(cache_key, data, timeout=3600)
        return data

    def get_image(self, obj):
        cache_key = f'route_image_{obj.id}'
        cached_image = cache.get(cache_key)
        if cached_image:
            return cached_image

        try:
            if obj.stops.exists() and obj.stops.first().images.exists():
                image_data = ThumbnailJSONSerializer(
                    instance=obj.stops.first().images.first().image, 
                    alias="", 
                    context=self.context
                ).data
            elif obj.cities.exists() and obj.cities.first().images.exists():
                image_data = ThumbnailJSONSerializer(
                    instance=obj.cities.first().images.first().image, 
                    alias="", 
                    context=self.context
                ).data
            else:
                image_data = None

            cache.set(cache_key, image_data, timeout=3600)
            return image_data
        except Exception as e:
            # Log the exception properly
            import logging
            logger = logging.getLogger(__name__)
            logger.error(f"Error fetching route image for route {obj.id}: {str(e)}")
            return None

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            for field in ['title']:
                if field in representation:
                    representation[field] = instance.display_text(field, language)
        
        return representation


class RouteListSerializer(BaseRouteSerializer):
    num_of_stops = serializers.SerializerMethodField(read_only=True)

    def get_num_of_stops(self, obj):
        cache_key = f'route_num_stops_{obj.id}'
        cached_count = cache.get(cache_key)
        if cached_count is not None:
            return cached_count
        
        stops_count = obj.stops.count()
        cache.set(cache_key, stops_count, timeout=3600)
        return stops_count

    class Meta:
        model = Route
        fields = (
            'id', 'title', 'tags', 'num_of_stops', 'num_of_views', 
            'is_top_ten', 'image', 'sort_order'
        )


class RouteListWithCitiesSerializer(RouteListSerializer):
    cities = serializers.SerializerMethodField()
    stops = serializers.SerializerMethodField()

    def get_cities(self, obj):
        cache_key = f'route_cities_{obj.id}'
        cached_cities = cache.get(cache_key)
        if cached_cities:
            return cached_cities
        
        cities_data = CitySerializer(
            obj.cities.all(), 
            many=True, 
            read_only=True, 
            context=self.context
        ).data
        cache.set(cache_key, cities_data, timeout=3600)
        return cities_data
    
    def get_stops(self, obj):
        return obj.stops.values_list('id', flat=True)

    class Meta:
        model = Route
        fields = (
            'id', 'title', 'tags', 'cities', 'num_of_stops', 
            'num_of_views', 'is_top_ten', 'image', 'sort_order', 'stops'
        )


class RouteStopSerializer(serializers.ModelSerializer):
    """Serializer for the RouteStop model to handle ordering."""
    
    class Meta:
        model = RouteStop
        fields = ('id', 'route', 'site', 'order')


class RouteDetailSerializer(RouteListSerializer):
    """Detailed route serializer with stops and cities."""
    cities = serializers.SerializerMethodField()
    stops = serializers.SerializerMethodField()

    def get_cities(self, obj):
        cache_key = f'route_cities_detail_{obj.id}'
        cached_cities = cache.get(cache_key)
        if cached_cities:
            return cached_cities
        
        cities_data = CitySerializer(
            obj.cities.all(), 
            many=True, 
            read_only=True, 
            context=self.context
        ).data
        cache.set(cache_key, cities_data, timeout=3600)
        return cities_data

    def get_stops(self, obj):
        cache_key = f'route_stops_{obj.id}_{self.context.get("request").headers.get("language", "en")}'
        cached_stops = cache.get(cache_key)
        if cached_stops:
            return cached_stops

        stops_data = SiteListSerializer(
            Site.objects.filter(route_stops__route__id=obj.id).order_by('route_stops__order'),
            many=True,
            read_only=True,
            context=self.context,
        ).data
        cache.set(cache_key, stops_data, timeout=3600)
        return stops_data

    class Meta:
        model = Route
        fields = (
            'id', 'title', 'tags', 'stops', 'num_of_stops', 
            'num_of_views', 'is_top_ten', 'cities', 'image', 'sort_order'
        )


class RouteSerializer(RouteDetailSerializer):
    """Main route serializer with full details."""
    pass


class CityWithRoutesSerializer(BaseCitySerializer):
    routes = serializers.SerializerMethodField()
    num_of_routes = serializers.SerializerMethodField()

    def get_num_of_routes(self, obj):
        cache_key = f'city_num_routes_{obj.id}'
        cached_count = cache.get(cache_key)
        if cached_count is not None:
            return cached_count
        
        routes_count = obj.routes.count()
        cache.set(cache_key, routes_count, timeout=3600)
        return routes_count

    def get_routes(self, obj):
        language = self.context.get('request').headers.get('language', 'en')
        cache_key = f'routes_for_city_{obj.id}_{language}'
        cached_routes = cache.get(cache_key)
        if cached_routes:
            return cached_routes

        output = RouteListSerializer(
            obj.routes.only('id', 'title', 'is_top_ten', 'num_of_views', 'sort_order'),
            context=self.context,
            many=True
        ).data
        cache.set(cache_key, output, timeout=3600)
        return output

    class Meta:
        model = City
        fields = ('id', 'name', 'routes', 'num_of_routes', 'slogan')


class RoutesAndSitesSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    title = serializers.CharField()
    images = serializers.SerializerMethodField(required=False)
    type = serializers.SerializerMethodField()
    city = CitySerializer(required=False)
    cities = CitySerializer(many=True, required=False)
    tags = TagOutputWithoutSelectionSerializer(many=True, read_only=True)
    num_of_stops = serializers.SerializerMethodField()
    stops = SiteWithoutCitySerializer(many=True, required=False)
    is_fav = serializers.BooleanField(required=False)
    address = AddressSerializer(required=False)
    levels = LevelListSerializer(many=True, required=False)
    categories = CategoryListSerializer(many=True, required=False)
    subcategories = SubCategorySerializer(many=True, required=False)
    sort_order = serializers.IntegerField(required=False)

    def get_images(self, obj):
        obj_type = 'route' if isinstance(obj, Route) else 'site'
        cache_key = f'{obj_type}_images_{obj.id}'
        cached_images = cache.get(cache_key)
        if cached_images:
            return cached_images

        if isinstance(obj, Route):
            if obj.stops.exists() and obj.stops.first().images.exists():
                images_data = ThumbnailJSONSerializer(
                    instance=obj.stops.first().images,
                    context=self.context,
                    alias='',
                    read_only=True,
                    many=True
                ).data
            else:
                images_data = []
        else:
            images_data = ThumbnailJSONSerializer(
                instance=obj.images,
                context=self.context,
                alias='',
                read_only=True,
                many=True
            ).data
        
        cache.set(cache_key, images_data, timeout=3600)
        return images_data

    def get_num_of_stops(self, obj):
        if isinstance(obj, Route):
            cache_key = f'route_num_stops_{obj.id}'
            cached_count = cache.get(cache_key)
            if cached_count is not None:
                return cached_count
            
            stops_count = obj.stops.count()
            cache.set(cache_key, stops_count, timeout=3600)
            return stops_count
        return 0

    def get_type(self, obj):
        return 'route' if isinstance(obj, Route) else obj.type

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            
            if isinstance(instance, Route):
                fields_to_translate = ['title']
            else:
                fields_to_translate = ['title', 'description']

            for field in fields_to_translate:
                if field in representation:
                    representation[field] = instance.display_text(field, language)
        
        return representation

    class Meta:
        fields = (
            'id', 'images', 'title', 'tags', 'num_of_stops', 'num_of_views',
            'is_top_ten', 'sort_order', 'type', 'cities', 'stops'
        )


# Route Order serializer for handling sorting
class RouteOrderSerializer(serializers.Serializer):
    route_id = serializers.IntegerField()
    order = serializers.IntegerField()

    class Meta:
        fields = ('route_id', 'order')
