"""Decorator that modifies the behavior of the function it wraps."""

from rest_framework.response import Response
from rest_framework import status
from local_secrets.service_integration.client import ResourceVerifySignature


def validate_signature(topic: str = None):
    """
    Custom decorator for Validating the request.
    """
    def decorator(func):
        def wrapper(self, request):

            if {'X-Signature', 'X-Timestamp'}.issubset(request.headers):
                signature = request.headers['X-Signature']
                timestamp = request.headers['X-Timestamp']

                # if no custom topic is used, then use the url name as the topic
                topic_name = topic if topic else request.resolver_match.url_name

                if not ResourceVerifySignature().validate_signature(
                        topic_name, timestamp, signature):
                    return Response({
                        "hasError": True,
                        "errorCode": 1002,
                        "message": "Error: Invalid signature.",
                    }, status=status.HTTP_403_FORBIDDEN)
            else:
                return Response({
                    "hasError": True,
                    "errorCode": 1002,
                    "message": "Error: Missing required headers.",

                }, status=status.HTTP_403_FORBIDDEN)
            return func(self, request)
        return wrapper
    return decorator


def language_required(func):
    """Validate language before accessing the data. """

    def wrapper(self, request):
        if not request.GET.get('language'):
            return Response({
                "hasError": True,
                "errorCode": 1002,
                "message": "Error: Missing or invalid 'language' parameter in the request.",
            })
        return func(self, request)
    return wrapper
