import random
import string
from io import BytesIO
from datetime import datetime, time
from collections import namedtuple
from typing import List, Tuple
from django.core.exceptions import MultipleObjectsReturned
import requests
from django.core.files.base import ContentFile
from PIL import Image as PILImage
from utils.s3_utils import S3
from config.settings import base
from local_secrets.sites.models import SiteImage, Schedule, HourRange, SiteType
from local_secrets.service_integration.exceptions import ImageDownloadException
from local_secrets.users.models import Tag, TranslatedTag
from local_secrets.languages.models import Language


class OperationTimings:
    """
    A utility class for managing and processing operating hours for a business.

    This class provides methods to:
    - Create and save operating hours for a business based on provided data.
    - Process individual timing slots for specific days and validate their format.
    """

    @staticmethod
    def _normalize_time(time_str: str):
        return time_str.upper().replace(
            "AM", " AM").replace(
            "PM", " PM").strip()

    def process_time_slots(self, day: str, timings: str, site_obj):
        """Processes timing slots for a given day and returns HourRange objects.
        """
        hour_objects = []
        schedule_obj = Schedule.objects.create(day=day, site=site_obj)
        timing_slot = timings.split(",")
        for timing in timing_slot:

            # Validate the operation hours
            initial_hour, end_hour = self.validate_operation_timing(timing)
            hour_objects.append(
                HourRange(
                    initial_hour=initial_hour,
                    end_hour=end_hour,
                    schedule=schedule_obj
                )
            )
        return hour_objects

    def validate_operation_timing(self, time_range: str):
        """
        Validates and converts a time range string (e.g., '7 AM–8 PM') into a
        tuple of 24-hour formatted times (e.g., ('07.00.00', '20.00.00')).
        """

        # Remove any non-breaking spaces and normalize input
        time_range = time_range.replace("\u202f", "").replace("–", "-").strip()

        # Special case: Open 24 hours
        if str(time_range).strip().lower() in ['open 24 hours', 'always open',
                                               'available 24 hours']:
            return time(0, 0), time(23, 59)

        if "-" not in time_range:
            raise ValueError(
                "Time range must contain an Hyphen (–) to separate start and end times."
            )

        start_time_str, end_time_str = time_range.split("-")

        # Normalize time strings by ensuring a space before AM/PM
        start_time_str = OperationTimings._normalize_time(start_time_str)
        end_time_str = OperationTimings._normalize_time(end_time_str)

        # Parse times with the correct format
        try:
            start_time = datetime.strptime(start_time_str, "%I:%M %p").time()
        except ValueError:
            start_time = datetime.strptime(start_time_str, "%I %p").time()

        try:
            end_time = datetime.strptime(end_time_str, "%I:%M %p").time()
        except ValueError:
            end_time = datetime.strptime(end_time_str, "%I %p").time()

        return start_time, end_time


class UploadSiteImages():
    """Transfer images from automation to local secret file system and digital ocean space.

    - Download google image urls.
    - Save originals image to local file system and digital ocean space.
    - Generate Thumbnails from each original image in to different sizes and
        save to the digital ocean space.
    """

    _bucket_name = base.AWS_STORAGE_BUCKET_NAME
    _path_prefix = "media/site_images"

    def __init__(self, image_urls: list, buisness_id: int):
        self.image_urls = image_urls
        self.buisness_id = buisness_id
        self.byte_streams = []
        self.s3_keys = []
        self.s3 = S3()
        self.image_formatter = {
            'thumbnail': (128, 128),
            'midsize': (720, 720),
            'fullsize': (1080, 1080),
        }

    def _generate_random_string(self, length=10):
        """
        Generates a random alphanumeric string followed by the current datetime to be used
        as a unique file name for saving to a digital ocean space.
        """
        date_time = datetime.now().strftime("%Y%m%d%H%M%S")
        characters = string.ascii_letters + string.digits
        initial_string = ''.join(random.choice(characters)
                                 for _ in range(length))
        final_string = initial_string + date_time
        return final_string

    def download_image(self):
        """
        Downloads an image from a given URL and returns the image as a BytesIO buffer.

        This function makes an HTTP request to download the image from the specified
        URL and stores it in memory using a BytesIO object. This allows the image to
        be accessed as a byte stream, which is useful for further processing or saving
        without needing to store it on disk.
        """
        for image_url in self.image_urls:
            try:
                response = requests.get(image_url, timeout=20)
                if response.status_code == 200:
                    buffer = BytesIO(response.content)
                    self.byte_streams.append(buffer)
                else:
                    raise ImageDownloadException(
                        message="Failed to download image.",
                        status_code=response.status_code,
                        url=image_url
                    )
            except requests.exceptions.HTTPError as http_err:
                raise requests.exceptions.HTTPError(
                    f"HTTP error occurred while downloading image from {image_url}: {http_err}"
                ) from http_err
            except requests.exceptions.RequestException as req_err:
                raise requests.exceptions.RequestException(
                    f"Request failed while downloading image from {image_url}: {req_err}"
                ) from req_err

    def upload_to_bucket(self, site_obj):
        """
        This function performs the following steps:
        1. Resizes the original image which was downloaded into three sizes: 128x128 (thumbnail),
            720x720 (medium), and 1080x1080 (large).
        2. Uploads the original image and resized images to the provided Digital ocean space.
        3. Optionally saves the original image as a binary field in the database.
        """
        for index, buffer in enumerate(self.byte_streams):
            try:
                # Generate a random string for file name.
                random_key = self._generate_random_string()
                filename = f'{random_key}.jpg'

                image = PILImage.open(buffer)
                original_buffer = BytesIO()
                image.save(original_buffer, format='JPEG')
                original_buffer.seek(0)

                # Save the original image to the local file system
                site_image_obj = SiteImage(site=site_obj)
                site_image_obj.image.save(
                    filename, ContentFile(original_buffer.read()),
                    save=False
                )

                # Upload them to the digital ocean
                self.s3.upload_to_s3(
                    self._bucket_name,
                    f"{self._path_prefix}/{filename}",
                    original_buffer
                )
                original_buffer.close()
                site_image_obj.save()

                # Generate thumbnails in different sizes
                thumbnail_sizes = [(128, 128), (720, 720), (1080, 1080)]
                for size in thumbnail_sizes:
                    resized_image = image.resize(size)

                    # Create a buffer for the resized image
                    resized_buffer = BytesIO()
                    resized_image.save(resized_buffer, format='PNG')
                    resized_buffer.seek(0)

                    # Create a filename for the resized image based on the size
                    filename = f"{filename}.{size[0]}x{size[1]}_q90.png"

                    # Upload them to the digital ocean
                    self.s3.upload_to_s3(
                        self._bucket_name,
                        f"{self._path_prefix}/{filename}",
                        resized_buffer,
                        'png'
                    )
                    resized_buffer.close()

            except Exception as e:
                raise Exception(
                    f"Error uploading image stream {index}: {str(e)}") from e


class QueryFilterGenerator:
    """
    A utility class for generating query filters and descriptions based on business data.
    """

    def __init__(self, request_data):
        self.business_data = request_data
        self.entity_filter = namedtuple(
            'FILTER', ['query_filter', 'description'])

    def _process_country(self):
        if country_id := self.business_data.get("country_id"):
            query_filter = {"id": country_id}
            filter_desc = f"id '{country_id}'"
        else:
            if country_data := self.business_data.get("country"):
                country_name = country_data.get("name")
                query_filter = {"name__iexact": country_name}
                filter_desc = f"name '{country_name}'"
            else:
                raise ValueError(
                    "Country details are missing for processing.")

        return self.entity_filter(
            query_filter=query_filter, description=filter_desc)

    def _process_city(self):
        if city_id := self.business_data.get("city_id"):
            query_filter = {"id": city_id}
            filter_desc = f"id '{city_id}'"
        else:
            if city_name := self.business_data.get("city"):
                query_filter = {"name__iexact": city_name}
                filter_desc = f"name '{city_name}'"
            else:
                raise ValueError(
                    "City details are missing for processing.")

        return self.entity_filter(
            query_filter=query_filter, description=filter_desc)

    def _process_level(self):
        if level_id := self.business_data.get("level_id"):
            query_filter = {"id": level_id}
            filter_desc = f"id '{level_id}'"
        else:
            query_filter = {
                "title__iexact": self.business_data.get("level"),
                "type": SiteType.PLACE
            }
            filter_desc = f"name '{self.business_data.get('level')}'"

        return self.entity_filter(
            query_filter=query_filter, description=filter_desc)

    def _process_category(self):
        # Initialize
        category_name = self.business_data.get("main_category")

        if category_id := self.business_data.get("category_id"):
            query_filter = {"id": category_id}
            filter_desc = f"id '{category_id}'"
        else:
            query_filter = {
                "title__iexact": category_name}
            filter_desc = f"name '{category_name}'"

        return self.entity_filter(
            query_filter=query_filter, description=filter_desc)

    def _process_sub_category(self, category_obj):

        # Initialize
        query_filter = None
        filter_desc = None
        sub_category_name = self.business_data.get("tailored_category")

        if sub_category_id := self.business_data.get("sub_category_id"):
            query_filter = {"id": sub_category_id}
            filter_desc = f"id '{sub_category_id}'"
        else:
            if sub_category_name:
                query_filter = {
                    "title__iexact": sub_category_name,
                    "category": category_obj.id
                }
                filter_desc = f"name '{sub_category_name}'"

        return self.entity_filter(
            query_filter=query_filter, description=filter_desc)

    def get_query_filter(self, entity_type: str, **kwargs):
        """Generate filter and description for the given entity type.
        """
        filter_mapping = {
            "country": self._process_country,
            "city": self._process_city,
            "level": self._process_level,
            "category": self._process_category,
            "subcategory": lambda: self._process_sub_category(
                kwargs.get("category_obj"))
        }
        method = filter_mapping.get(entity_type)
        if not method:
            raise ValueError("Invalid mapping.")

        return method()


class TagLink:
    """
    A utility class for transforming the tags to a specified format and store dats in to 
    Tags and Translated tags table.
    """

    def __init__(self, buisness_data: dict):
        self.buisness_data = buisness_data
        self.types_esp = self.buisness_data.get("types_esp")

    def _reformat_tags(self) -> List[Tuple[str, str, str, str]]:
        """
        Convert the string values into lists and pair corresponding translations like
        [('spanish value', 'french value', 'english uk value', 'english us value'), ].
        """
        try:
            if self.types_esp:
                tag_data = {
                    "types_esp": self.types_esp,  # default
                    "types_fr": self.buisness_data.get("types_fr"),  # french
                    "types_eng": self.buisness_data.get("types_eng"),  # uk
                    "types": self.buisness_data.get("types")  # us
                }

                # Find the count of tags inorder to process further
                tag_length = len(
                    [x.strip() for x in str(tag_data["types_esp"]).split(",")]
                )

                tag_list = [
                    [
                        x.strip()
                        for x in tag.split(",")
                    ]
                    # if the selected tag not exist ensure it matches other lists length
                    if tag else [""] * tag_length
                    for tag in tag_data.values()
                ]

                # Uses `zip(*)` to pair corresponding tags from all languages into tuples.
                return list(zip(*tag_list))
            else:
                raise ValueError(
                    "Tag not found. A valid tag is required to proceed.")

        except Exception as e:
            raise

    def assign(self) -> List:
        """
        Stores the  parent tag (in Spanish) in Tag table and its corresponding translations in French, 
        English (US), and English (UK) in the translated tag table. 
        """
        parent_tags = []  # storing parent tags (in spanish)

        # Reformat the tags
        tag_collection = self._reformat_tags()
        try:
            for tags in tag_collection:
                spanish_title = tags[0]
                translated_data = {
                    'fr': tags[1],
                    'en-GB': tags[2],
                    'en': tags[3]
                }

                try:
                    parent_obj, _ = Tag.objects.get_or_create(
                        title__iexact=spanish_title,
                        defaults={"title": spanish_title}
                    )
                except MultipleObjectsReturned:
                    parent_obj = Tag.objects.filter(
                        title__iexact=spanish_title).last()

                parent_tags.append(parent_obj)

                # Process translated tags
                for lang_code, title_value in translated_data.items():
                    if title_value:
                        try:
                            TranslatedTag.objects.get_or_create(
                                title__iexact=title_value, tag=parent_obj,
                                language=Language.objects.filter(
                                    code=lang_code).last(),
                                defaults={"title": title_value})
                        except MultipleObjectsReturned:
                            pass

            return parent_tags

        except Exception as e:
            raise
