# local_secrets/sites/analytics.py

from functools import cache
from django.db.models import Count, Avg, Q, Sum, F, ExpressionWrapper, fields
from django.db.models.functions import Extract
 
from django.utils import timezone
from datetime import timedelta 
from local_secrets.sites.choices import SiteStatusChoices
from local_secrets.sites.models import Site, SiteReview, SiteViewLog
from local_secrets.users.models import VendorSubscription

class SiteAnalytics:
    def __init__(self, date_range=30):
        self.date_range = date_range
        self.start_date = timezone.now() - timedelta(days=date_range)
  
    def _get_pending_reviews_count(self):
        return Site.objects.filter(
            status=SiteStatusChoices.PENDING_REVIEW
        ).count()

    def _get_average_review_time(self):
        completed_reviews = SiteReview.objects.filter(
            created_at__gte=self.start_date,
            status__in=[SiteStatusChoices.APPROVED, SiteStatusChoices.REJECTED]
        )
        
        if not completed_reviews.exists():
            return None
            
        return completed_reviews.aggregate(
            avg_time=Avg('updated_at') - Avg('created_at')
        )['avg_time']

    def _get_approval_rate(self):
        total_reviews = SiteReview.objects.filter(
            created_at__gte=self.start_date
        ).count()
        
        if not total_reviews:
            return 0
            
        approved_reviews = SiteReview.objects.filter(
            created_at__gte=self.start_date,
            status=SiteStatusChoices.APPROVED
        ).count()
        
        return (approved_reviews / total_reviews) * 100

    def _get_review_status_distribution(self):
        return SiteReview.objects.filter(
            created_at__gte=self.start_date
        ).values('status').annotate(
            count=Count('id')
        )

    def _get_active_subscriptions_count(self):
        return VendorSubscription.objects.filter(
            status='active',
            end_date__gte=timezone.now().date()
        ).count()

    def _get_subscription_plan_distribution(self):
        return VendorSubscription.objects.filter(
            status='active'
        ).values('plan__name').annotate(
            count=Count('id')
        )

    def _get_revenue_metrics(self):
        subscriptions = VendorSubscription.objects.filter(
            created_at__gte=self.start_date
        )
        
        return {
            'total_revenue': subscriptions.aggregate(
                total=Sum('plan__price')
            )['total'] or 0,
            'new_subscriptions': subscriptions.count(),
            'renewal_rate': self._calculate_renewal_rate()
        }
 
    def get_view_analytics(self, site_id=None):
        """Get comprehensive view analytics"""
        view_tracker = ViewTrackingAnalytics(site_id)
        
        return {
            'daily_metrics': view_tracker.get_view_metrics('day'),
            'weekly_metrics': view_tracker.get_view_metrics('week'),
            'monthly_metrics': view_tracker.get_view_metrics('month'),
            'yearly_metrics': view_tracker.get_view_metrics('year')
        }


    def get_review_metrics(self):
        """Get review-related metrics"""
        reviews = SiteReview.objects.filter(
            created_at__gte=self.start_date 
        )
        
        # Calculate average review time in hours
        review_time = reviews.annotate(
            duration=ExpressionWrapper(
                ExpressionWrapper(
                    (Extract(F('updated_at'), 'epoch') - Extract(F('created_at'), 'epoch')) / 3600,
                    output_field=fields.FloatField()
                ),
                output_field=fields.FloatField()
            )
        ).aggregate(
            avg_time=Avg('duration')
        )['avg_time'] or 0


        total_reviews = reviews.count()
        approved_reviews = reviews.filter(status='APPROVED').count()
        approval_rate = (approved_reviews / total_reviews * 100) if total_reviews > 0 else 0


        return {
            'pending_reviews': reviews.filter(status='PENDING').count(),
            'average_review_time': round(review_time, 2),
            'approval_rate': round(approval_rate, 2),
            'total_reviews': total_reviews
        }


    def get_subscription_metrics(self):
        """Get subscription-related metrics"""
        from local_secrets.users.models import VendorSubscription
        
        subscriptions = VendorSubscription.objects.filter(
            start_date__gte=self.start_date
        )
        
        active_subs = subscriptions.filter(
            status='active',
            end_date__gt=timezone.now()
        )
        
        total_revenue = active_subs.aggregate(
            total=Sum('plan__price')
        )['total'] or 0


        return {
            'active_subscriptions': active_subs.count(),
            'total_revenue': total_revenue,
            'new_subscriptions': subscriptions.count(),
            'renewal_rate': self._calculate_renewal_rate()
        }


    def get_view_metrics(self):
        """Get view-related metrics"""
        views = SiteViewLog.objects.filter(
            viewed_at__gte=self.start_date
        )
        
        return {
            'total_views': views.count(),
            'unique_visitors': views.values('ip_address').distinct().count(),
            'popular_hours': self._get_popular_hours(views),
            'view_trend': self._get_view_trend(views)
        }


    def get_top_viewed_sites(self, limit=10):
        """Get top viewed sites"""
        return SiteViewLog.objects.filter(
            viewed_at__gte=self.start_date
        ).values(
            'site__id',
            'site__title'
        ).annotate(
            view_count=Count('id'),
            unique_visitors=Count('ip_address', distinct=True)
        ).order_by('-view_count')[:limit]


    def get_view_trends(self):
        """Get view trends by day"""
        return SiteViewLog.objects.filter(
            viewed_at__gte=self.start_date
        ).extra(
            select={'date': 'DATE(viewed_at)'}
        ).values('date').annotate(
            views=Count('id'),
            unique_visitors=Count('ip_address', distinct=True)
        ).order_by('date')


    def _calculate_renewal_rate(self):
        """Calculate subscription renewal rate"""
        from local_secrets.users.models import VendorSubscription
        
        # Find subscriptions that ended in our date range
        expired_subs = VendorSubscription.objects.filter(
            end_date__range=[self.start_date, timezone.now()]
        )
        
        if not expired_subs.exists():
            return 0
        
        # Count how many of these vendors have a new subscription
        renewed_count = expired_subs.filter(
            vendor__subscriptions__start_date__gt=F('end_date')
        ).distinct().count()
        
        return round((renewed_count / expired_subs.count()) * 100, 2)


    def _get_popular_hours(self, views):
        """Get popular hours for site views"""
        return views.extra(
            select={'hour': 'EXTRACT(HOUR FROM viewed_at)'}
        ).values('hour').annotate(
            count=Count('id')
        ).order_by('-count')[:5]


    def _get_view_trend(self, views):
        """Get view trend by hour"""
        return views.extra(
            select={'hour': 'EXTRACT(HOUR FROM viewed_at)'}
        ).values('hour').annotate(
            count=Count('id')
        ).order_by('hour')


class ViewTrackingAnalytics:
    """Handles view tracking analytics separately for better organization"""
    def __init__(self, site_id=None):
        self.site_id = site_id
        
    def track_view(self, user_id=None, ip_address=None):
        """Track a new view with additional metadata"""
        from local_secrets.sites.models import SiteViewLog
        
        view_data = {
            'site_id': self.site_id,
            'viewed_at': timezone.now(),
            'ip_address': ip_address
        }
        
        if user_id:
            view_data['user_id'] = user_id
            
        SiteViewLog.objects.create(**view_data)
        
        # Update cache for real-time analytics
        self._update_view_cache()
        
    def get_view_metrics(self, period='day'):
        """Get view metrics for different time periods"""
        cache_key = f'site_views_{self.site_id}_{period}'
        cached_data = cache.get(cache_key)
        
        if cached_data:
            return cached_data
            
        metrics = self._calculate_view_metrics(period)
        cache.set(cache_key, metrics, timeout=3600)  # Cache for 1 hour
        
        return metrics
        
    def _calculate_view_metrics(self, period):
        """Calculate view metrics for a specific period"""
        from local_secrets.sites.models import SiteViewLog
        
        now = timezone.now()
        
        if period == 'day':
            start_date = now - timedelta(days=1)
        elif period == 'week':
            start_date = now - timedelta(weeks=1)
        elif period == 'month':
            start_date = now - timedelta(days=30)
        else:
            start_date = now - timedelta(days=365)
            
        views = SiteViewLog.objects.filter(
            site_id=self.site_id,
            viewed_at__gte=start_date
        )
        
        return {
            'total_views': views.count(),
            'unique_visitors': views.values('ip_address').distinct().count(),
            'authenticated_views': views.exclude(user_id=None).count(),
            'peak_hour': self._get_peak_hour(views),
            'view_trend': self._get_view_trend(views, period)
        }
        
    def _get_peak_hour(self, views):
        """Calculate the hour with most views"""
        return views.annotate(
            hour=F('viewed_at__hour')
        ).values('hour').annotate(
            count=Count('id')
        ).order_by('-count').first()
        
    def _get_view_trend(self, views, period):
        """Calculate view trend over time"""
        if period == 'day':
            return views.annotate(
                interval=F('viewed_at__hour')
            )
        elif period == 'week':
            return views.annotate(
                interval=F('viewed_at__date')
            )
        else:
            return views.annotate(
                interval=F('viewed_at__week')
            )
            
    def _update_view_cache(self):
        """Update real-time view cache"""
        cache_key = f'site_views_realtime_{self.site_id}'
        current_hour = timezone.now().replace(minute=0, second=0, microsecond=0)
        
        views_data = cache.get(cache_key, {})
        views_data[current_hour.isoformat()] = views_data.get(current_hour.isoformat(), 0) + 1
        
        # Keep only last 24 hours
        cutoff = (current_hour - timedelta(hours=24)).isoformat()
        views_data = {k: v for k, v in views_data.items() if k >= cutoff}
        
        cache.set(cache_key, views_data, timeout=86400)  # Cache for 24 hours


