
# local_secrets/sites/api_views.py  

from django.urls import reverse
from rest_framework import viewsets
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import AllowAny
from django.db.models import Q, Prefetch
import unicodedata

from local_secrets.cities.models import Country, Region, County, City, Neighborhood
from local_secrets.sites.models import Category, Site, SiteType, SubCategory
from local_secrets.cities.serializers import (
    CountrySerializer, RegionSerializer, CountySerializer, 
    CitySerializer, NeighborhoodSerializer
) 
from rest_framework.views import APIView
 
from local_secrets.sites.models import Site, SiteType
from local_secrets.sites.serializers import   SiteListSerializer, SiteSerializer 


def remove_accent_marks(text):
    """
    Remove accent marks from input text.
    """
    normalized = unicodedata.normalize('NFD', text)
    return ''.join(c for c in normalized if not unicodedata.category(c).startswith('M'))


class UnifiedSearchViewSet(viewsets.GenericViewSet):
    """
    ViewSet providing unified search functionality across cities and countries with proper translation support.
    """
    permission_classes = []

    @action(detail=False, methods=['get'])
    def search(self, request):
        """
        Search for cities and countries with a keyword, supporting translated names.

        Query parameters:
        - keyword: Text to search for cities and countries.
        - types: Comma-separated list of entity types to include (cities, countries, or all).
        - limit: Maximum number of results per entity type (default: 5).
        - include_coordinates: Boolean to include latitude and longitude for cities.
        """
        keyword = request.query_params.get("keyword", "").strip()
        types = request.query_params.get("types", "all").lower().split(",")
        limit = int(request.query_params.get("limit", 5))
        include_coordinates = request.query_params.get("include_coordinates", "true").lower() == "true"
        lang = request.query_params.get("language", "en")  # Default language to English

        if not keyword:
            return Response({"detail": "Keyword parameter is required"}, status=400)

        _keyword = remove_accent_marks(keyword.lower())
        has_accents = keyword.lower() != _keyword

        results = {} 
        
        # Search Cities
        if "all" in types or "cities" in types:
            cities_query = Q(name__icontains=keyword)

            if has_accents:
                cities_query |= Q(search_name__icontains=_keyword)

            # Add translation support
            if lang:
                cities_query |= Q(translations__language__code=lang, translations__name__icontains=keyword)

            cities = City.objects.filter(cities_query).distinct()[:limit]

            city_data = CitySerializer(cities, many=True, context={"request": request, "lang": lang}).data

            # Add `type` and construct accessible URL for each city
            if include_coordinates:
                for city_item, city_instance in zip(city_data, cities):
                    city_item["latitude"] = city_instance.latitude if hasattr(city_instance, "latitude") else None
                    city_item["longitude"] = city_instance.longitude if hasattr(city_instance, "longitude") else None
                    city_item["type"] = "city"
                    city_item["id"] = city_instance.id  # Add ID explicitly
                    city_item["url"] = request.build_absolute_uri(f"/cities/{city_instance.id}/")

                    # Add country information
                    if hasattr(city_instance, "country") and city_instance.country:
                        city_item["country_id"] = city_instance.country.id
                        city_item["country_name"] = self._get_translated_name(city_instance.country, lang)

            results["cities"] = city_data
            
        # Search Countries
        if "all" in types or "countries" in types:
            countries_query = Q(name__icontains=keyword)
            if has_accents:
                countries_query |= Q(search_name__icontains=_keyword)
            
            # Add translation support
            if lang:
                countries_query |= Q(translations__language__code=lang, translations__name__icontains=keyword)
            
            countries = Country.objects.filter(countries_query).distinct()[:limit]
            country_data = CountrySerializer(countries, many=True, context={"request": request, "lang": lang}).data
            
            # Add additional information
            for country_item, country_instance in zip(country_data, countries):
                country_item["code"] = country_instance.code if hasattr(country_instance, 'code') else None
                country_item["continent"] = country_instance.continent if hasattr(country_instance, 'continent') else None
                country_item["cities_count"] = country_instance.cities.count() if hasattr(country_instance, 'cities') else 0
            
            results["countries"] = country_data

                # Search Neighborhoods
        if "all" in types or "neighborhoods" in types:
            neighborhoods_query = Q(name__icontains=keyword)
            if has_accents:
                neighborhoods_query |= Q(search_name__icontains=_keyword)

            neighborhoods = Neighborhood.objects.filter(neighborhoods_query)[:limit]
            results["neighborhoods"] = NeighborhoodSerializer(neighborhoods, many=True, context={"request": request}).data

        # Search Regions
        if "all" in types or "regions" in types:
            regions_query = Q(name__icontains=keyword)
            if has_accents:
                regions_query |= Q(search_name__icontains=_keyword)

            regions = Region.objects.filter(regions_query)[:limit]
            results["regions"] = RegionSerializer(regions, many=True, context={"request": request}).data

        # Search Events and Places (Sites)
        if "all" in types or "places" in types or "events" in types:
            sites_query = Q(title__icontains=keyword)
            if has_accents:
                sites_query |= Q(search_name__icontains=_keyword)

            base_query = Site.objects.filter(sites_query).prefetch_related("address", "categories")

            if "all" in types:
                events = base_query.filter(type=SiteType.EVENT)[:limit]
                places = base_query.filter(type=SiteType.PLACE)[:limit]

                results["events"] = self._add_coordinates_to_sites(
                    SiteListSerializer(events, many=True, context={"request": request}).data, events
                )
                results["places"] = self._add_coordinates_to_sites(
                    SiteListSerializer(places, many=True, context={"request": request}).data, places
                )
            else:
                if "events" in types:
                    events = base_query.filter(type=SiteType.EVENT)[:limit]
                    results["events"] = self._add_coordinates_to_sites(
                        SiteListSerializer(events, many=True, context={"request": request}).data, events
                    )
                if "places" in types:
                    places = base_query.filter(type=SiteType.PLACE)[:limit]
                    results["places"] = self._add_coordinates_to_sites(
                        SiteListSerializer(places, many=True, context={"request": request}).data, places
                    )


        return Response(results)
    
    def _get_translated_name(self, obj, lang=None):
        """
        Get the translated name of an object based on the language.
        
        Args:
            obj: The object to get the translated name from.
            lang: The language code.
            
        Returns:
            The translated name if available, otherwise the default name.
        """
        if not lang or not hasattr(obj, 'translations'):
            return obj.name
            
        translation = obj.translations.filter(language__code=lang).first()
        return translation.name if translation else obj.name
    
    def _add_coordinates_to_sites(self, serialized_data, site_instances):
        """
        Add latitude and longitude coordinates to serialized site data.

        Args:
            serialized_data: List of serialized site data.
            site_instances: Queryset of Site instances.

        Returns:
            List of serialized site data with latitude and longitude included (if available).
        """
        for site_data, site_instance in zip(serialized_data, site_instances):
            latitude = None
            longitude = None

            if hasattr(site_instance, 'address') and site_instance.address:
                address = site_instance.address
                if hasattr(address, 'latitude') and hasattr(address, 'longitude'):
                    latitude = address.latitude
                    longitude = address.longitude
                elif hasattr(address, 'location') and address.location:
                    latitude = address.location.y
                    longitude = address.location.x

            site_data["latitude"] = latitude
            site_data["longitude"] = longitude

        return serialized_data


class SiteSearchView(APIView):
    """
    View providing search functionality for the Site model, including translations for categories, subcategories, cities, and extended results.
    """

    def get(self, request, search_type, keyword):
        """
        Handles GET requests for searching places or events, including translations for related fields and additional data like city ID, type, image, etc.
        """
        if not keyword:
            return Response({"error": "Keyword is required"}, status=400)

        # Normalize keyword to handle accents
        _keyword = remove_accent_marks(keyword.lower())

        # Get language parameter (default to 'en' if not provided)
        language = request.query_params.get('language', 'en')

        # Build query to search for title and its translations
        query = Q(title__icontains=keyword) | Q(title__icontains=_keyword)
        query |= Q(translations__title__icontains=keyword, translations__language__code=language)
        query |= Q(translations__title__icontains=_keyword, translations__language__code=language)

        # Filter based on the search type
        if search_type == "place":
            query &= Q(type=SiteType.PLACE)
        elif search_type == "event":
            query &= Q(type=SiteType.EVENT)
        else:
            return Response({"error": "Invalid search type"}, status=400)

        # Retrieve sites matching the query with a limit of 10
        sites = Site.objects.prefetch_related(
            Prefetch('categories', queryset=Category.objects.prefetch_related('translations')),
            Prefetch('subcategories', queryset=SubCategory.objects.prefetch_related('translations')),
            Prefetch('city', queryset=City.objects.prefetch_related('translations')),
            'address',
            'translations'
        ).filter(query).distinct()[:10]

        serializer = SiteListSerializer(sites, many=True, context={"request": request})

        # Build the response, including translations for related fields and additional data
        results = []

        for site_instance, site_data in zip(sites, serializer.data):
            # Fetch categories and subcategories with translations
            category_titles = [
                category.translations.filter(language__code=language).first().title
                if category.translations.filter(language__code=language).exists()
                else category.title
                for category in site_instance.categories.all()
            ]

            subcategory_titles = [
                subcategory.translations.filter(language__code=language).first().title
                if subcategory.translations.filter(language__code=language).exists()
                else subcategory.title
                for subcategory in site_instance.subcategories.all()
            ]

            # Get city name with translations and city ID
            if site_instance.city:
                city_translation = site_instance.city.translations.filter(
                    language__code=language).first()
                city_name = city_translation.name if city_translation else site_instance.city.name
                city_id = site_instance.city.id
            else:
                city_name = "Unknown"
                city_id = None

            # Get coordinates
            latitude = None
            longitude = None

            # Option 1: Coordinates stored in Address model
            if site_instance.address:
                if hasattr(site_instance.address, 'latitude') and hasattr(site_instance.address, 'longitude'):
                    latitude = site_instance.address.latitude
                    longitude = site_instance.address.longitude
                elif hasattr(site_instance.address, 'location') and site_instance.address.location:
                    location = site_instance.address.location
                    try:
                        latitude = location.y
                        longitude = location.x
                    except AttributeError:
                        pass
                elif hasattr(site_instance.address, 'coordinates') and site_instance.address.coordinates:
                    try:
                        latitude, longitude = site_instance.address.coordinates
                    except (ValueError, TypeError):
                        pass

            # Option 2: Coordinates stored directly in Site model
            elif hasattr(site_instance, 'latitude') and hasattr(site_instance, 'longitude'):
                latitude = site_instance.latitude
                longitude = site_instance.longitude
            elif hasattr(site_instance, 'location') and site_instance.location:
                try:
                    latitude = site_instance.location.y
                    longitude = site_instance.location.x
                except AttributeError:
                    pass

            # Get translated site title or fallback to the original title
            translated_title = (
                site_instance.translations.filter(language__code=language).first().title
                if site_instance.translations.filter(language__code=language).exists()
                else site_data["title"]
            )

            # Append results
            results.append({
                "id": site_data["id"],
                "title": translated_title,
                "categories": category_titles,
                "subcategories": subcategory_titles,
                "city": city_name,
                "city_id": city_id,
                "type": site_instance.type,  # Include type (e.g., place or event)
                "image": site_data.get("image"),  # Include the image field if available
                "rating": site_data.get("rating", "No Rating"),
                "latitude": latitude,
                "longitude": longitude,
                "description": site_data.get("description"),  # Include description if available
                "created_at": site_data.get("created_at"),  # Include created_at timestamp for tracking
                "updated_at": site_data.get("updated_at"),  # Include updated_at timestamp
            })

        return Response(results)


    
# --- Additional Documentation in README.md or docs/api.md ---

"""
# Unified Search API

## Overview
The Unified Search API allows searching across multiple entity types (cities, countries, sites, etc.) 
with a single request. It's useful for implementing unified search bars or general search functionality.

## Endpoint
GET /sites/unified-search/search/

## Query Parameters
- `keyword` (required): Text to search for across all entities
- `types` (optional): Comma-separated list of entity types to include in the search
  - Default: 'all'
  - Available options: 'countries', 'regions', 'counties', 'cities', 'neighborhoods', 'events', 'places', 'sites'
- `limit` (optional): Maximum number of results per entity type
  - Default: 5

## Headers
- `language` (optional): Language code for localized results (e.g., 'es', 'en')

## Example Requests

### Search across all entity types
GET /sites/unified-search/search/?keyword=Dominican

### Search only in countries and cities
GET /sites/unified-search/search/?keyword=Dominican&types=countries,cities

### Search events with a higher limit
GET /sites/unified-search/search/?keyword=Festival&types=events&limit=10

## Example Response
{
  "countries": [
    {
      "id": 1,
      "name": "Dominican Republic",
      "code": "DO"
    }
  ],
  "cities": [
    {
      "id": 45,
      "name": "Santo Domingo",
      "country": "Dominican Republic"
    }
  ],
  "sites": [
    {
      "id": 12,
      "name": "Dominican Fiesta Hotel",
      "type": "PLACE"
    },
    {
      "id": 23,
      "name": "Dominican Festival 2023",
      "type": "EVENT"
    }
  ]
}
"""
# --- End of Documentation ---