from datetime import datetime, timedelta

from django.contrib.gis.geos import Point, MultiPoint, LineString, MultiLineString, Polygon
from django.db import models
from django.db.models import Case, Count, Exists, F, OuterRef, Q, Subquery, When, BooleanField, Value, IntegerField, \
    Max, ExpressionWrapper, DateField, DurationField
from django.db.models.functions import Abs, Coalesce, Cast
from django.forms import CharField
from django.utils.timezone import now, localtime
from rest_framework.exceptions import ValidationError
from django.db.models import Avg, Value, FloatField
from django.db.models.functions import Coalesce
from decimal import Decimal
from local_secrets.core.utils.text import TextHelper as text
from local_secrets.languages.models import Language
from local_secrets.sites.choices import Day, SiteStatusChoices, SiteSubscriptionStatus
from decimal import Decimal, InvalidOperation
from django.db.models import Q
 

class SiteManager(models.Manager):
    def get_queryset(self):
        return (
            super(SiteManager, self).get_queryset().filter(
                city__activated=True, has_been_accepted=True
            )
            .filter(
                Q(type='place') |
                Q(frequency__in=['day', 'workday', 'week', 'year'], type='event') |
                Q(special_schedules__day__gte=now().date(), type='event', frequency='never') |
                Q(activated=True) 
            ) 
        )


class AdminSiteManager(models.Manager):
    def get_queryset(self):
        return super(AdminSiteManager, self).get_queryset().annotate_translated_fields()
 
class SiteManager(models.Manager):
    def get_queryset(self):
        return SiteQuerySet(self.model, using=self._db)

    def with_translations(self, language=None):
        return self.get_queryset().annotate_translated_fields(language)

class SiteQuerySet(models.QuerySet):
    def get(self, *args, **kwargs):
        try:
            return super(SiteQuerySet, self).get(*args, **kwargs)
        except self.model.MultipleObjectsReturned:
            return super(SiteQuerySet, self).filter(id=kwargs.get('id')).first()

    def create(self, **kwargs):
        levels = None
        categories = None
        subcategories = None
        schedules = None

        if 'levels' in kwargs:
            levels = kwargs.pop('levels')

        if 'categories' in kwargs:
            categories = kwargs.pop('categories')

        if 'subcategories' in kwargs:
            subcategories = kwargs.pop('subcategories')

        if 'schedules' in kwargs:
            schedules = kwargs.pop('schedules')

        site = super(SiteQuerySet, self).create(**kwargs)

        if levels:
            site.levels.set(levels)

        if categories:
            site.categories.set(categories)

        if subcategories:
            site.subcategories.set(subcategories)

        if schedules:
            from local_secrets.sites.models import HourRange, Schedule

            for schedule_json in schedules:
                schedule = Schedule.objects.create(day=schedule_json.get('day'), site=site)
                for opening_hours in schedule_json.get('opening_hours'):
                    HourRange.objects.create(
                        initial_hour=opening_hours.get('initial_hour'),
                        end_hour=opening_hours.get('end_hour'),
                        schedule=schedule,
                    )
                schedule.save()

        return site

    def favorites(self, user):
        return user.fav_sites.all()

    def filter_type(self, type):
        if type:
            return self.filter(type=type)
        return self

    def filter_keyword(self, keyword, language='es'):
        """
        Filter queryset by keyword. Skip translations for default Spanish.
        """
        if keyword:
            keyword = keyword.strip().lower()
            _keyword = text.remove_all_accent_marks(keyword)
            has_accents = keyword != _keyword

            if language == 'es':  # Default Spanish
                query = Q(title__icontains=keyword) if has_accents else Q(title__unaccent__icontains=_keyword)
            else:  # Other languages use translations
                query = Q(
                    translations__title__icontains=keyword
                    if has_accents
                    else Q(translations__title__unaccent__icontains=_keyword)
                )

            return self.filter(query).distinct()
        return self

    def filter_levels(self, levels):
        sites = self
        if levels:
            if type(levels) == list:
                sites = sites.filter(levels__id__in=levels)
            else:
                splitted_levels_ids = levels.split(',')
                if len(splitted_levels_ids) > 1:
                    sites = sites.filter(levels__id__in=splitted_levels_ids)
                else:
                    sites = sites.filter(levels__id=levels)
        return sites

    def filter_categories(self, categories):
        sites = self
        if categories:
            if type(categories) == list:
                sites = sites.filter(categories__id__in=categories)
            else:
                splitted_categories_ids = categories.split(',')
                if len(splitted_categories_ids) > 1:
                    sites = sites.filter(categories__id__in=splitted_categories_ids)
                else:
                    sites = sites.filter(categories__id=categories)
        return sites

    def filter_subcategories(self, subcategories):
        sites = self
        if subcategories:
            if type(subcategories) == list:
                sites = sites.filter(subcategories__id__in=subcategories)
            else:
                splitted_subcategories_ids = subcategories.split(',')
                if len(splitted_subcategories_ids) > 1:
                    sites = sites.filter(subcategories__id__in=splitted_subcategories_ids)
                else:
                    sites = sites.filter(subcategories__id=subcategories)
        return sites

    def filter_hours(self, day, hour):
        sites = self
        if day:
            sites = sites.filter(schedules__day=day)
        if hour:
            try:
                hour = datetime.strptime(hour, '%H:%M')
            except ValueError:
                raise ValidationError(detail='The hour does not have a valid format', code=400)
            sites = sites.filter(
                schedules__opening_hours__initial_hour__lte=hour, schedules__opening_hours__end_hour__gte=hour
            )
        return sites

    def filter_special_schedule(self, first_date, last_date):
        sites = self
        if first_date:
            sites = sites.filter(special_schedules__day__gte=first_date)
        if last_date:
            sites = sites.filter(special_schedules__day__lte=last_date)
        return sites

    def filter_suggested(self, is_suggested):
        sites = self
        if is_suggested:
            sites = sites.filter(is_suggested=True)
        return sites

    def filter_city(self, city, language='es'):
        """
        Filters sites by city name considering translations and unaccented versions.
        """
        if not city:
            return self

        city = city.strip().lower()
        _city = text.remove_all_accent_marks(city)  # Remove accents
        has_accents = city != _city  # Check for accents

        if language == 'es':  # Default to non-translated Spanish
            query = Q(city__name__icontains=city) if has_accents else Q(city__name__unaccent__icontains=_city)
        else:  # Use translated fields for other languages
            query = Q(
                city__translations__name__icontains=city if has_accents
                else Q(city__translations__name__unaccent__icontains=_city)
            )

        return self.filter(query).distinct()

    def filter_city_id(self, city_id):
        sites = self
        if city_id:
            if type(city_id) == list:
                sites = sites.filter(city__id__in=city_id)
            else:
                splitted_cities_ids = city_id.split(',')
                if len(splitted_cities_ids) > 1:
                    sites = sites.filter(city__id__in=splitted_cities_ids)
                else:
                    sites = sites.filter(city__id=city_id)
        return sites

    def filter_tag(self, tag):
        sites = self
        if tag:
            sites = sites.filter(tags__id=tag)
        return sites

    def filter_country(self, country, language='es'):
        """
        Filters sites based on the country name while considering translations.
        """
        if not country:
            return self

        # Normalize input country name
        country = country.strip().lower()
        _country = text.remove_all_accent_marks(country)
        has_accents = country != _country

        # Default Spanish handling
        if language == 'es':
            query = Q(city__country__name__icontains=country) if has_accents else Q(city__country__name__unaccent__icontains=_country)
        else:
            # Check translations
            query = (
                Q(city__country__translations__name__icontains=country) |
                Q(city__country__name__icontains=country)
                if has_accents else
                Q(city__country__translations__name__unaccent__icontains=_country) |
                Q(city__country__name__unaccent__icontains=_country)
            )
            # Fallback to original data when translation is missing
            query |= Q(city__country__name__icontains=country)

        return self.filter(query).distinct()

    def filter_country_id(self, country_id):
        sites = self
        if country_id:
            sites = sites.filter(city__country__id=country_id)
        return sites

    def filter_by_datetime(self, current_datetime):
        sites = self
        if current_datetime:
            sites = sites.annotate(
                union=Case(When(schedules__opening_hours__initial_hour__gt=F('schedules__opening_hours__end_hour')))
            ).filter(
                Q(
                    schedules__opening_hours__initial__hour__lte=current_datetime,
                    schedules__opening_hours__end_hour__gte=current_datetime,
                    union=True,
                )
                | Q(
                    schedules__opening_hours__initial__hour__gte=current_datetime,
                    schedules__opening_hours__end_hour__lte=current_datetime,
                    union=False,
                )
            )
        return sites

    def filter_avg_rating(self, rating: str = None, is_sort_by: bool = False):
            """
            Filters Site objects based on custom rating categories received from the frontend.

            Custom rating categories and their corresponding filtering conditions:

            - 4 (Excellent): Sites with an average rating of 4 or more.
            - 3 (Very Good): Sites with an average rating of 3 or more.
            - 2 (Good): Sites with an average rating of 2 or more.
            - 1 (Room for Improvement): Sites with an average rating of less than 2.
            """
            sites = self
            if rating:
                query = Q()
                try:
                    # Handle range (e.g., "4.9-5.0")
                    if '-' in rating:
                        start, end = map(str.strip, rating.split('-'))
                        start = Decimal(start)
                        end = Decimal(end)
                        query = Q(average_rating__gte=start, average_rating__lte=end)
                    else:
                        # Handle single value (e.g., "4.5")
                        query = Q(average_rating__gte=Decimal(rating))
                except (InvalidOperation, ValueError) as e:
                    raise ValueError(f"Invalid rating format: {rating}") from e
                # Annotate and filter based on the query
                sites = sites.annotate(
                    average_rating=Coalesce(
                        Avg('comments__rating', output_field=models.DecimalField()),
                        Value(Decimal('0.00'), output_field=models.DecimalField())
                    )
                ).filter(query)
            elif is_sort_by:
                # Sort by average rating if sorting is enabled
                sites = sites.annotate(
                    average_rating=Coalesce(
                        Avg('comments__rating', output_field=models.DecimalField()),
                        Value(Decimal('0.00'), output_field=models.DecimalField())
                    )
                )
            return sites
        
    def annotate_num_of_favs(self):
            return self.annotate(num_of_favs=Count('users'))

    def annotate_is_fav(self, user):
            if user.is_anonymous:
                return self
            return self.annotate(is_fav=Exists(Subquery(user.fav_sites.filter(id=OuterRef('id')))))

    def get_translation_annotation(field_name, language_code):
        from local_secrets.sites.models import TranslatedSite
        """Dynamically create Subqueries for translated fields."""
        return Subquery(
            TranslatedSite.objects.filter(
                site__id=OuterRef('pk'),
                language__code=language_code
            ).values(field_name)[:1]
        )

    def annotate_all_translations(self):
        """Annotate all translated fields for each language."""
        queryset = self
        from local_secrets.sites.models import Language  # Import Language model
        languages = Language.objects.all()
        
        for language in languages:
            queryset = queryset.annotate(
                **{
                    f'translated_title_{language.code}': self.get_translation_annotation('title', language.code),
                    f'translated_description_{language.code}': self.get_translation_annotation('description', language.code),
                }
            )
        return queryset

    def annotate_language_specific_fields(self, language_code):
        """Annotate fields for a specified language, with fallback to defaults."""
        if not language_code or language_code.lower() == 'es':  # Default Spanish
            return self.annotate(
                translated_title=F('title'),
                translated_description=F('description'),
            )

        # Annotate translations for specified language, with fallback
        return self.annotate(
            translated_title=Coalesce(
                self.get_translation_annotation('title', language_code),
                F('title'),  # Fallback to default title
                output_field=models.CharField()
            ),
            translated_description=Coalesce(
                self.get_translation_annotation('description', language_code),
                F('description'),  # Fallback to default description
                output_field=models.CharField()
            )
        ).filter(
            Q(translations__language__code=language_code) | Q(translations__isnull=True)
        ).distinct()

    def active_subscriptions(self):
        """Filter sites with active subscriptions"""
        return self.filter(subscription_status=SiteSubscriptionStatus.ACTIVE)

    def publishable(self):
        """Filter sites that can be published"""
        return self.filter(
            status=SiteStatusChoices.APPROVED,
            subscription_status=SiteSubscriptionStatus.ACTIVE
        )
 
    def filter_type(self, type_name):
        if type_name:
            return self.filter(type=type_name.lower())
        return self


    def filter_keyword(self, keyword, language='es'):
        if keyword:
            print(keyword)
            keyword = keyword.lower().rstrip(" ").lstrip(" ")
            _keyword = text.remove_all_accent_marks(keyword)
            has_accents = keyword != _keyword
            print(_keyword)
            if language == 'es':
                if has_accents:
                    title_query = Q(title__icontains=keyword)
                else:
                    title_query = Q(title__unaccent__icontains=_keyword)
                # description_query = Q(description__icontains=keyword)
            elif language != 'es':
                if has_accents:
                    title_query = Q(translations__title__icontains=keyword)
                else:
                    title_query = Q(translations__title__unaccent__icontains=_keyword)
                # description_query = Q(translations__description__icontains=keyword)
            else:
                if has_accents:
                    title_query = Q(title__icontains=keyword)
                else:
                    title_query = Q(title__unaccent__icontains=_keyword)
                # description_query = Q(description__icontains=keyword)
            return self.filter(title_query)  # | description_query)
        return self


    def filter_polygon(self, points):
        if points and None not in points:
            print(points)
            mp = MultiPoint(
                *map(lambda x: Point(float(x.split(',')[1]), float(x.split(',')[0]), srid=4326), points),
                srid=4326
            )
            print(mp.coords)
            coords = list(mp.coords)
            coords.append(coords[0])
            polygon = Polygon(tuple(coords))
            return self.filter(address__point__within=polygon)
        else:
            return self

    def annotate_is_open(self):
        current_date = localtime()
        current_weekday = current_date.weekday() + 1  # Django utiliza 1 para domingo y 7 para sábado
        days = {2: 'monday', 3: 'tuesday', 4: 'wednesday', 5: 'thursday', 6: 'friday', 7: 'saturday', 1: 'sunday'}
        current_weekday_name = days[(current_weekday + 1) % 7]
        current_day = current_date.day

        return self.annotate(is_open=Case(
            When(always_open=True, then=Value(True, output_field=BooleanField())),
            When(frequency='never', then=Value(False, output_field=BooleanField())),
            When(frequency='day', then=Value(True, output_field=BooleanField())),
            When(
                frequency='week',
                then=Case(
                    When(schedules__day=current_weekday_name, then=Value(True)),
                    default=Value(False, output_field=BooleanField())
                )
            ),
            When(
                frequency='month',
                then=Case(
                    When(special_schedules__day__day=current_day, then=Value(True)),
                    default=Value(False, output_field=BooleanField())
                )
            ),
            When(
                frequency='year',
                then=Case(
                    When(special_schedules__day=current_date.date(), then=Value(True)),
                    default=Value(False, output_field=BooleanField())
                )
            ),
            When(
                frequency='workday',
                then=Value(current_weekday < 6, output_field=BooleanField())
            ),
            default=Value(False, output_field=BooleanField()),
        )
        )

    def annotate_translated_fields(self, language="es"):
        """
        Annotates translated fields for the given language.
        """
        from local_secrets.sites.models import TranslatedSite
        if not language or language == 'es':  # Default primary fields if language is Spanish or missing
            return self.annotate(
                translated_title=F('title'),
                translated_description=F('description'),
            )
        
        # Properly handle language as a string literal for non-default languages
        return self.annotate(
            translated_title=Coalesce(
                Subquery(
                    TranslatedSite.objects.filter(
                        site_id=OuterRef("id"),
                        language_id=Subquery(
                            Language.objects.filter(code=str(language)).values('id')[:1]
                        )
                    ).values("title")[:1]
                ),
                F("title"),  # Fallback to default
                output_field=models.CharField(),
            ),
            translated_description=Coalesce(
                Subquery(
                    TranslatedSite.objects.filter(
                        site_id=OuterRef("id"),
                        language_id=Subquery(
                            Language.objects.filter(code=str(language)).values('id')[:1]
                        )
                    ).values("description")[:1]
                ),
                F("description"),  # Fallback to default
                output_field=models.CharField(),
            )
        )
 

    def annotate_num_of_favs(self):
        return self.annotate(num_of_favs=models.Count('users'))

    def annotate_is_fav(self, user):
        from local_secrets.sites.models import FavoriteSites
        if not user or user.is_anonymous:
            return self.annotate(is_fav=Value(False, output_field=models.BooleanField()))
        return self.annotate(
            is_fav=models.Exists(
                FavoriteSites.objects.filter(
                    site_id=models.OuterRef('pk'),
                    user=user
                )
            )
        )

    def with_next_schedule_date(self):
        current_time = localtime()

        # Annotate `day_order` based on `current_time.weekday()` for each schedule
        annotated_schedules = self.annotate(
            day_order=Case(
                *[When(schedules__day=day, then=Value(index)) for index, day in
                  enumerate(["monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday"])],
                output_field=IntegerField(),
            )
        )

        # Annotate `day_distance` using calculated `day_order` as a base for each Site instance
        annotated_sites = annotated_schedules.annotate(
            day_distance=Case(
                When(day_order__gte=current_time.weekday(), then=F('day_order') - (current_time.weekday())),
                When(day_order__lt=current_time.weekday(), then=7 - (current_time.weekday() - F('day_order'))),
                output_field=IntegerField()
            ),
            next_date=ExpressionWrapper(
                Cast(Value(current_time), DateField()) + Cast(F("day_distance"), output_field=IntegerField()),
                output_field=DateField()
            ),
            closest_day=Max('special_schedules__day')
        )

        # Order the schedules by `day_distance` to get the closest one
        #closest_schedule = annotated_sites.order_by('day_distance').first()

        # If there's a match, calculate the date of the next schedule
        #next_schedule_date = current_time + timedelta(days=closest_schedule.day_distance) if closest_schedule and closest_schedule.day_distance else None
        #next_schedule_date = closest_schedule.day_order if closest_schedule and closest_schedule.day_order else closest_schedule.closest_day

        return annotated_sites
 
    def active_subscriptions(self):
        """Filter sites with active subscriptions"""
        return self.filter(subscription_status=SiteSubscriptionStatus.ACTIVE)

    def publishable(self):
        """Filter sites that can be published"""
        return self.filter(
            status=SiteStatusChoices.APPROVED,
            subscription_status=SiteSubscriptionStatus.ACTIVE
        )

    def annotate_day_order(self, current_weekday):
        """Annotate day order for scheduling"""
        return self.annotate(
            day_order=Case(
                *[When(day=day, then=Value(i)) for i, day in enumerate(Day.values)],
                output_field=IntegerField(),
            ),
            day_distance=Case(
                When(day_order__gte=current_weekday, 
                     then=F('day_order') - current_weekday),
                When(day_order__lt=current_weekday, 
                     then=F('day_order') - current_weekday + 7),
                output_field=IntegerField(),
            )
        )


class ScheduleQuerySet(models.QuerySet):
    def annotate_day_order(self, current_day_integer):
        return self.annotate(
            day_order=Case(
                When(day=Day.MONDAY, then=0),
                When(day=Day.TUESDAY, then=1),
                When(day=Day.WEDNESDAY, then=2),
                When(day=Day.THURSDAY, then=3),
                When(day=Day.FRIDAY, then=4),
                When(day=Day.SATURDAY, then=5),
                When(day=Day.SUNDAY, then=6),
                output_field=models.IntegerField(),
                default=-1,
            ),
            day_distance=Case(
                When(day_order__lt=current_day_integer, then=7 - (current_day_integer - F('day_order'))),
                When(day_order__gte=current_day_integer, then=Abs(current_day_integer - F('day_order'))),
            ),
        )


class SpecialScheduleQuerySet(models.QuerySet):
    def annotate_day_distance(self, current_day):
        return self.annotate(day_distance=current_day - F('day'))


class BaseManager(models.Manager):  # Base manager for Level, Category and SubCategory models
    def get_queryset(self):
        return super(BaseManager, self).get_queryset().order_by('order')


class LevelManager(models.Manager):
    def get_queryset(self):
        return (super(LevelManager, self).get_queryset()).annotate(
            site_count=Count('sites', output_field=IntegerField())
        ).filter(site_count__gt=0).order_by('order')


class CategoryManager(models.Manager):
    def get_queryset(self):
        return (super(CategoryManager, self).get_queryset()).annotate(
            site_count=Count('sites', output_field=IntegerField())
        ).filter(site_count__gt=0).order_by('order')


class SubcategoryManager(models.Manager):
    def get_queryset(self):
        return (super(SubcategoryManager, self).get_queryset()).annotate(
            site_count=Count('sites', output_field=IntegerField())
        ).filter(site_count__gt=0).order_by('order')

class CommentQuerySet(models.QuerySet):
    def filter_rating(self, rating):
        comments = self

        if rating:
            query = Q()
            splitted_rating_ids = rating.split(',')
            for rating in splitted_rating_ids:
                rating = int(rating)
                if rating == 1:
                    query |= Q(rating__lt=2)
                else:
                    query |= Q(rating__gte=rating)
            
            comments = comments.filter(query)
        return comments
    
    def filter_language(self, language):
        comments = self
        if language:
           pass
        return comments
    



