# local_secrets/sites/middleware.py 

from django.utils import timezone
from django.core.cache import cache
from datetime import timedelta
from django.db import models

from local_secrets.sites.analytics import ViewTrackingAnalytics


class SiteViewMiddleware:
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        response = self.get_response(request)
        
        if (
            response.status_code == 200 and 
            request.resolver_match and 
            request.resolver_match.url_name in ['site-detail', 'event-detail']
        ):
            site_id = request.resolver_match.kwargs.get('pk')
            if site_id:
                user_id = request.user.id if request.user.is_authenticated else None
                ip_address = self.get_client_ip(request)
                
                tracker = ViewTrackingAnalytics(site_id)
                tracker.track_view(user_id, ip_address)
                
        return response



    def process_site_view(self, request, site_id):
        """Process a site view with rate limiting"""
        ip_address = self.get_client_ip(request)
        user_id = request.user.id if request.user.is_authenticated else None
        
        cache_keys = self.generate_cache_keys(site_id, ip_address, user_id)

        if self.should_count_view(cache_keys):
            # Use atomic increment
            from django.db import transaction
            from local_secrets.sites.models import Site
            
            with transaction.atomic():
                Site.objects.filter(id=site_id).update(
                    view_count=models.F('view_count') + 1
                ) 
            self.set_view_cache(cache_keys)


    def get_client_ip(self, request):
        x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
        if x_forwarded_for:
            return x_forwarded_for.split(',')[0]
        return request.META.get('REMOTE_ADDR')


    def generate_cache_keys(self, site_id, ip_address, user_id):
        """Generate cache keys for rate limiting"""
        timestamp = timezone.now().strftime('%Y%m%d%H')
        keys = [
            f'site_view:{site_id}:ip:{ip_address}:{timestamp}',
        ]
        if user_id:
            keys.append(f'site_view:{site_id}:user:{user_id}:{timestamp}')
        return keys


    def should_count_view(self, cache_keys):
        """Check if the view should be counted"""
        return not any(cache.get(key) for key in cache_keys)


    def set_view_cache(self, cache_keys):
        """Set cache to prevent duplicate counts"""
        for key in cache_keys:
            cache.set(key, True, timeout=3600)  # 1 hour timeout
