# local_secrets/sites/models.py

from datetime import datetime, timedelta
from decimal import Decimal
from django.conf import settings
from django.utils import timezone
from django.contrib.auth import get_user_model
from django.contrib.gis.geos import Point, MultiPoint, Polygon
from django.core.cache import cache
from django.core.validators import FileExtensionValidator, MaxValueValidator, MinValueValidator
from django.db import models
from django.db.models import F, Q, OuterRef, Subquery, Value, BooleanField, Avg
from django.utils.timezone import localtime, now
from django.utils.translation import gettext_lazy as _
from django.utils.text import slugify
from easy_thumbnails.fields import ThumbnailerImageField
from django.core.exceptions import ValidationError
from local_secrets.cities.models import Address, City
from local_secrets.core.custom_exceptions import CustomApiException
from local_secrets.languages.models import Language
from local_secrets.sites.choices import Day, FrequencyChoices, SiteSubscriptionStatus, SiteType, SiteStatusChoices, SiteVisibilityChoices
from django.core.exceptions import PermissionDenied
from local_secrets.sites.managers import (
    ScheduleQuerySet, 
    SiteManager,
    SiteQuerySet, 
    SpecialScheduleQuerySet, 
    LevelManager,
    CategoryManager,
    SubcategoryManager,
    BaseManager, 
    CommentQuerySet,
    AdminSiteManager
)
from local_secrets.sites.state_machine import StateTransitionError
from local_secrets.users.models import CustomUser, Tag
from django.db.models import Count, F, Q, Exists, OuterRef, Subquery
from django.utils.timezone import now


class SiteVisibilityState(models.TextChoices):
    """
    Represents the visibility states of a site
    Controls site visibility based on subscription and approval status
    """
    DRAFT = 'draft', _('Draft')  # Initial state when site is created
    PENDING_REVIEW = 'pending_review', _('Pending Review')  # Submitted for approval
    APPROVED = 'approved', _('Approved')  # Admin approved, waiting for subscription
    ACTIVE = 'active', _('Active')  # Approved and subscription active
    INACTIVE = 'inactive', _('Inactive')  # Subscription expired/cancelled
    SUSPENDED = 'suspended', _('Suspended')  # Manually suspended by admin
    ARCHIVED = 'archived', _('Archived')  # Archived by vendor

class SubscriptionState(models.TextChoices):
    """
    Represents the subscription states
    Controls feature access and quota limitations
    """
    NONE = 'none', _('No Subscription')  # Initial state
    TRIAL = 'trial', _('Trial')  # Trial period
    ACTIVE = 'active', _('Active')  # Paid and active
    PAST_DUE = 'past_due', _('Past Due')  # Payment failed but grace period
    CANCELLED = 'cancelled', _('Cancelled')  # Cancelled by vendor
    EXPIRED = 'expired', _('Expired')  # Subscription expired
    SUSPENDED = 'suspended', _('Suspended')  # Suspended due to violation

class SiteQuerySet(models.QuerySet):
    def filter_keyword(self, keyword, language=None):
        if not keyword:
            return self
        if language:
            return self.filter(
                Q(translations__title__icontains=keyword) |
                Q(translations__description__icontains=keyword)
            )
        return self.filter(
            Q(title__icontains=keyword) |
            Q(description__icontains=keyword)
        )

    def filter_categories(self, categories):
        if categories:
            return self.filter(categories__id__in=categories.split(','))
        return self

    def filter_levels(self, levels):
        if levels:
            return self.filter(levels__id__in=levels.split(','))
        return self

    def filter_subcategories(self, subcategories):
        if subcategories:
            return self.filter(subcategories__id__in=subcategories.split(','))
        return self

    def filter_city(self, city, language=None):
        if not city:
            return self
        if language:
            return self.filter(city__translations__name__icontains=city)
        return self.filter(city__name__icontains=city)

    def filter_city_id(self, city_id):
        if city_id:
            return self.filter(city_id=city_id)
        return self

    def filter_tag(self, tag):
        if tag:
            return self.filter(tags__id=tag)
        return self

    def filter_country(self, country):
        if country:
            return self.filter(city__country__name__icontains=country)
        return self

    def filter_hours(self, day, hour):
        if not (day and hour):
            return self
        return self.filter(
            Q(schedules__day=day) &
            Q(schedules__opening_hours__initial_hour__lte=hour) &
            Q(schedules__opening_hours__end_hour__gte=hour)
        )

    def filter_suggested(self, is_suggested):
        if is_suggested is not None:
            return self.filter(is_suggested=is_suggested)
        return self

    def filter_special_schedule(self, first_date, last_date):
        if first_date and last_date:
            return self.filter(
                special_schedules__day__range=[first_date, last_date]
            )
        return self

    def filter_polygon(self, points):
        if points and None not in points:
            print(points)
            mp = MultiPoint(
                *map(lambda x: Point(float(x.split(',')[1]), float(x.split(',')[0]), srid=4326), points),
                srid=4326
            )
            print(mp.coords)
            coords = list(mp.coords)
            coords.append(coords[0])
            polygon = Polygon(tuple(coords))
            return self.filter(address__point__within=polygon)
        else:
            return self

    def annotate_is_fav(self, user):
        if not user or user.is_anonymous:
            return self.annotate(is_favorite=Value(False, output_field=BooleanField()))
        return self.annotate(
            is_favorite=Exists(
                FavoriteSites.objects.filter(
                    site_id=OuterRef('pk'),
                    user=user
                )
            )
        )
 
class AdminSiteManager(models.Manager):
    def get_queryset(self):
        return SiteQuerySet(self.model, using=self._db)
 
class SiteSubscriptionQuerySet(models.QuerySet):
    def active_subscriptions(self):
        return self.filter(subscription_status=SiteSubscriptionStatus.ACTIVE)

    def publishable(self):
        return self.filter(
            status=SiteStatusChoices.APPROVED,
            subscription_status=SiteSubscriptionStatus.ACTIVE
        )
 
class Site(models.Model):
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Title'))
    slug = models.SlugField(
        max_length=500, 
        blank=True,
        null=True,
        db_index=True,
        verbose_name=_('URL Slug')
    )
    levels = models.ManyToManyField('Level', verbose_name=_('Levels'), related_name='sites')
    categories = models.ManyToManyField('Category', verbose_name=_('Categories'), related_name='sites')
    subcategories = models.ManyToManyField('SubCategory', blank=True, verbose_name=_('Subcategories'), related_name='sites')
    type = models.CharField(max_length=100, choices=SiteType.choices, default=SiteType.PLACE, verbose_name=_('Type'))
    description = models.TextField(verbose_name=_('Description'))
    status = models.CharField(max_length=20,choices=SiteStatusChoices.choices, default=SiteStatusChoices.DRAFT,db_index=True)
    is_suggested = models.BooleanField(default=False)
    has_been_accepted = models.BooleanField(default=True)
    frequency = models.CharField(max_length=100, choices=FrequencyChoices.choices, default='never', verbose_name=_('Frequency'))
    media = models.FileField(upload_to='site_videos',null=True,blank=True,validators=[FileExtensionValidator(allowed_extensions=['mp4', 'avi', 'mov', 'webm', 'mkv'])],verbose_name=_('Video'),)
    url = models.CharField(max_length=500, verbose_name=_('Link'), null=True, blank=True)
    phone = models.CharField(max_length=500, verbose_name=_('Contact phone'), null=True, blank=True)
    tags = models.ManyToManyField(Tag, related_name='site_tags', verbose_name=_('Tags'))
    users = models.ManyToManyField(CustomUser, related_name='fav_sites', through='FavoriteSites', verbose_name=_('Users'))
    address = models.ForeignKey(Address, on_delete=models.SET_NULL, related_name='sites', null=True, blank=True)
    city = models.ForeignKey(City, on_delete=models.SET_NULL, related_name='sites', null=True, blank=True, verbose_name=_('City'))
    created_by = models.ForeignKey(CustomUser, on_delete=models.SET_NULL, null=True, verbose_name=_('Created by'))
    always_open = models.BooleanField(default=False, verbose_name=_('Always open'))
    is_top_10 = models.BooleanField(default=False, verbose_name=_('Is top 10'))
    activated = models.BooleanField(default=False)
    vendor = models.ForeignKey('users.Vendor',on_delete=models.SET_NULL,null=True, blank=True,related_name='vendor_sites',verbose_name=_('Vendor'))
    is_vendor_featured = models.BooleanField(default=False, verbose_name=_('Vendor Featured'))
    vendor_notes = models.TextField(blank=True, verbose_name=_('Vendor Notes'))
    vendor_contact_info = models.CharField(max_length=500, blank=True, verbose_name=_('Vendor Contact Info'))
    view_count = models.PositiveIntegerField(default=0, verbose_name=_('View Count'))
    review_feedback = models.TextField(verbose_name=_('Review Feedback'), null=True, blank=True)
    reviewed_by = models.ForeignKey('users.CustomUser', on_delete=models.SET_NULL, null=True, blank=True, related_name='reviewed_sites')
    reviewed_at = models.DateTimeField(null=True, blank=True)
    subscription_status = models.CharField(max_length=20,choices=SiteSubscriptionStatus.choices,default=SiteSubscriptionStatus.FREE,db_index=True)
    active_subscription = models.ForeignKey('users.VendorSubscription',on_delete=models.SET_NULL,null=True,blank=True,related_name='active_sites')
    
    is_visible = models.BooleanField(default=False, help_text=_('Indicates if the site is publicly visible'))
    visibility_changed_at = models.DateTimeField(null=True, blank=True) 
    visibility = models.CharField(max_length=20,choices=SiteVisibilityChoices.choices,default=SiteVisibilityChoices.HIDDEN,db_index=True)
    visibility_update_pending = models.BooleanField(default=False)
    
    visibility_state = models.CharField(max_length=20, choices=SiteVisibilityState.choices, default=SiteVisibilityState.DRAFT, db_index=True)
    subscription_state = models.CharField(max_length=20, choices=SubscriptionState.choices, default=SubscriptionState.NONE, db_index=True)

    state_changed_at = models.DateTimeField(auto_now=True)
    state_changed_by = models.ForeignKey(
        settings.AUTH_USER_MODEL,  
        on_delete=models.SET_NULL,
        null=True,
        related_name='site_state_changes'
    )
    state_change_reason = models.CharField(max_length=255, blank=True)
    price = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True, verbose_name=_('Regular Price')) 
    update_attempts = models.IntegerField(default=0)
    last_update_attempt = models.DateTimeField(null=True)   
    last_status_change = models.DateTimeField(auto_now=True)
    status_change_reason = models.CharField(max_length=100, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    published_at = models.DateTimeField(null=True, blank=True)
    unpublished_at = models.DateTimeField(null=True, blank=True)
 
    objects = SiteManager.from_queryset(SiteQuerySet)()
    objects_for_admin = AdminSiteManager.from_queryset(SiteQuerySet)() 
  
    @property
    def active_promotion(self):
        """Returns the current active promotion for this site"""
        current_time = timezone.now()
        return self.site_promotions.filter(
            is_active=True,
            start_date__lte=current_time,
            end_date__gte=current_time
        ).select_related('vendor').first()

    @property
    def discounted_price(self):
        """Calculate discounted price if promotion exists"""
        if not self.price:
            return None
            
        promotion = self.active_promotion
        if not promotion:
            return self.price
            
        if promotion.discount_type == 'percentage':
            discount = (promotion.discount_value / 100) * self.price
            return max(0, self.price - discount)
        return max(0, self.price - promotion.discount_value)
    
    def can_have_promotion(self):
        """Check if site can have promotions"""
        return (
            self.subscription_status == SiteSubscriptionStatus.ACTIVE and
            self.status == SiteStatusChoices.PUBLISHED and
            self.visibility == SiteVisibilityChoices.VISIBLE and
            self.vendor is not None and
            self.vendor.get_active_subscription().plan.can_create_promotions
        )

    def increment_view_count(self):
        """
        Increment view count using F() to prevent race conditions
        """
        from django.db import transaction
        with transaction.atomic():
            self.__class__.objects.filter(id=self.id).update(
                view_count=models.F('view_count') + 1
            )
        cache.delete(f'site_view_count_{self.id}')
        self.refresh_from_db(fields=['view_count'])
        
        return self.view_count

    def change_state(self, new_state, user):
        if not user.is_staff and not user.is_superuser:
            raise PermissionDenied(_("Only admin users can change site status"))
        
        self.state = new_state
        self.state_changed_by = user
        self.save(update_fields=['state', 'state_changed_by'])
        
        return {
            'status': 'success',
            'message': _('Site state updated successfully'),
            'new_state': new_state
        }
    
    @property
    def total_views(self):
        """ Get cached view count """
        cache_key = f'site_view_count_{self.id}'
        count = cache.get(cache_key)        
        if count is None:
            count = self.view_count
            cache.set(cache_key, count, timeout=3600)  # Cache for 1 hour
            
        return count
    
    def is_vendor_managed(self):
        """Check if this site is managed by a vendor"""
        return self.vendor is not None
    
    def get_vendor_analytics(self):
        """Get analytics for this site that are useful for the vendor"""
        return {
            'view_count': self.view_count,
            'favorite_count': self.users.count(),
            'comments_count': getattr(self, 'comments', []).count() if hasattr(self, 'comments') else 0,
        }
        
    def transfer_to_vendor(self, vendor):
        """Transfer ownership of this site to a vendor"""
        if not self.vendor:
            self.vendor = vendor
            self.save(update_fields=['vendor'])
            return True
        return False

    def is_open_now(self, current_time=now()):
        """Check if the site is currently open, considering vendor hours if applicable"""
        if self.always_open:
            return True
            
        # If the site has a vendor, check vendor opening hours first
        if self.vendor:
            day_of_week = current_time.weekday()  # 0 = Monday, 6 = Sunday
            current_time_only = current_time.time() 
            vendor_hours = self.vendor.opening_hours.filter(day_of_week=day_of_week)
            
            for hours in vendor_hours:
                if hours.from_hour <= current_time_only <= hours.to_hour:
                    return True 
            if vendor_hours.exists():
                return False 
        return super().is_open_now(current_time)

    def display_text(self, field, language='en'):
        # Try fetching the specified translated field
        translation = self.translations.filter(language__code=language).first()
        if translation:
            return getattr(translation, field)
        # Fallback to the default language
        translation = self.translations.filter(language__code='es').first()
        if translation:
            return getattr(translation, field)
        # Fallback to the base model field
        return getattr(self, field)

    def mark_as_fav(self, user):
        fav_site, created = FavoriteSites.objects.get_or_create(site=self, user=user)
        if not created:
            fav_site.delete()
        if created and user.fav_sites.count() > 50:
            fav_site.delete()
            raise CustomApiException(message=_('El usuario ya tiene más de 50 favoritos'), status_code=406)
        return created

    def is_open_by_schedule(self, current_time=now()):
        if self.always_open:
            return True
        days = dict(zip(range(0, 6), Day.choices))
        if self.type == SiteType.PLACE:  # The place Site uses only regular schedules from Mon-Sun.
            try:
                schedule = self.schedules.prefetch_related('opening_hours').get(day=days[current_time.weekday()][0])
                use_union = schedule.opening_hours.filter(initial_hour__gt=F('end_hour')).exists()
            except Schedule.DoesNotExist:
                print('There is no schedule for the specified day')
                return False
            except Exception:
                return False
        else:  # The event Site uses special schedules
            try:
                schedule = self.special_schedules.get(day=current_time.date())
                use_union = schedule.opening_hours.filter(initial_hour__gt=F('end_hour')).exists()
            except SpecialSchedule.DoesNotExist:
                print('There is no special schedule for the specified day')
                return False

        if not use_union:
            time_ranges = schedule.opening_hours.filter(
                initial_hour__lte=current_time.time(), end_hour__gte=current_time.time()
            )
        else:
            time_ranges = schedule.opening_hours.filter(
                Q(initial_hour__lte=current_time.time()) | Q(end_hour__gte=current_time.time())
            )
        return time_ranges.exists()

    def next_schedule(self):
        current_time = localtime()
        if self.type == SiteType.PLACE:
            schedule = self.schedules.annotate_day_order(current_time.weekday()).order_by('day_distance').first()
            return schedule
        else:
            if self.frequency in ['week', 'day', 'workday']:
                # This is meant to return a SpecialSchedule with the next week's day given the Schedule,
                # not the SpecialSchedule of the event.

                # Get the closest Schedule on the event
                next_schedule = self.schedules.annotate_day_order(current_time.weekday()).order_by('day_distance').first()
                if next_schedule:
                    # Calculate the difference for the next day on next_schedule on the next week
                    if next_schedule.day_order >= current_time.date().weekday():
                        # If the day is after today's weekday
                        days_until_next = next_schedule.day_order - current_time.date().weekday()
                    else:
                        # If the next_schedule's weekday is before or equal to today
                        days_until_next = 7 - (current_time.date().weekday() - next_schedule.day_order)
                    if days_until_next == 7:
                        next_schedule_datetime = current_time
                    else:
                        next_schedule_datetime = current_time + timedelta(days=days_until_next)
                    special_schedule = SpecialSchedule(
                        site=self,
                        day=next_schedule_datetime.date()
                    )
                    return special_schedule
                else:
                    return None

                # return self.schedules.annotate_day_order(current_time.weekday()).order_by('day_distance').first()
            else:
                special_schedules = self.special_schedules.annotate_day_distance(current_time.date())
                schedule = special_schedules.order_by('day_distance').first()
                if self.frequency in ['year',] and (schedule and schedule.day < current_time.date()):
                    schedule = special_schedules.order_by('-day_distance').first()
                    schedule = SpecialSchedule(
                        id=self.id,
                        site=self,
                        day=schedule.day.replace(year=current_time.date().year + 1),   # Add one year
                    )
                return schedule

    def check_frequency(self, is_open):
        current_date = localtime()

        if self.frequency == 'never':
            print(type(is_open))
            return True if is_open else False
        elif self.frequency == 'day':
            return True
        elif self.frequency == 'week':
            # Use the normal Schedule to check if today is applicable
            days = dict(zip(range(0, 6), Day.choices))
            res = self.schedules.filter(day=days[current_date.date().weekday()][0]).exists()
            return res

            # Old way of using the special_schedules
            # weekday = self.special_schedules.first().day.weekday()
            # if current_date.date().weekday() == weekday:
                # return True
        elif self.frequency == 'month':
            month_day = self.special_schedules.first().day
            if current_date.date().day == month_day:
                return True
        elif self.frequency == 'year':
            day = self.special_schedules.first().day
            if current_date.date() == day:
                return True
        elif self.frequency == 'workday':
            if current_date.weekday() < 5:
                return True
        return False

    def open_days(self):
        special_schedules = self.special_schedules.order_by('day')
        return special_schedules.first(), special_schedules.last()

    def add_comment(self, user, body, rating):
        comment = Comment.objects.create(user=user, site=self, body=body, rating=rating, created_at=now())
        return comment

    def create_address(self, location):
        geolocation = location.get('geolocation').split(',')
        point = Point((Decimal(geolocation[0]), Decimal(geolocation[1])), srid=4326)
        city = City.objects.filter(**location.get('city')).first()
        if city is None:
            self.save()
            return
        address = Address.objects.create(
            street=location.get('street_name'), city=city, point=point, details=location.get('link')
        )
        self.address = address
        self.city = city
        self.save()

    def add_schedules(self, schedules):
        for schedule in schedules:
            day = schedule.get('day')
            for hours in schedule.get('opening_hours'):
                initial_hour = hours.get('initial_hour')
                end_hour = hours.get('end_hour')

                db_schedule = Schedule.objects.create(site=self, day=day)
                HourRange.objects.create(initial_hour=initial_hour, end_hour=end_hour, schedule=db_schedule)

    def add_special_schedules(self, special_schedules):
        for special_schedule in special_schedules:
            initial_date = special_schedule.get('initial_date')
            end_date = special_schedule.get('end_date')
            initial_hour = special_schedule.get('initial_hour')
            end_hour = special_schedule.get('end_hour')

            date_format = '%Y-%m-%d'
            try:
                datetime.strptime(initial_date, date_format)
            except ValueError:
                date_format = '%I:%M %p'

            all_days = [
                datetime.strptime(initial_date, date_format).date() + timedelta(days=x)
                for x in range(
                    (
                        (datetime.strptime(end_date, date_format).date() + timedelta(days=1))
                        - datetime.strptime(initial_date, date_format).date()
                    ).days
                )
            ]
            for day in all_days:
                db_special_schedule = SpecialSchedule.objects.create(day=day, site=self)
                SpecialHourRange.objects.create(
                    schedule=db_special_schedule, initial_hour=initial_hour, end_hour=end_hour
                )

    def submit_for_review(self):
        """Submit site for admin review"""
        self.status = SiteStatusChoices.PENDING_REVIEW
        self.save()

    def approve_site(self, admin_user, feedback=None):
        """Approve site after review"""
        self.status = SiteStatusChoices.APPROVED
        self.reviewed_by = admin_user
        self.reviewed_at = timezone.now()
        self.review_feedback = feedback
        self.save()

    def reject_site(self, admin_user, feedback):
        """Reject site with feedback"""
        self.status = SiteStatusChoices.REJECTED
        self.reviewed_by = admin_user
        self.reviewed_at = timezone.now()
        self.review_feedback = feedback
        self.save()

    def update_subscription_status(self):
        """Update site subscription status based on active subscription"""
        if not self.active_subscription:
            new_status = SiteSubscriptionStatus.FREE
        elif not self.active_subscription.is_active():
            new_status = SiteSubscriptionStatus.EXPIRED
        else:
            new_status = SiteSubscriptionStatus.ACTIVE
        
        if new_status != self.subscription_status:
            self.subscription_status = new_status
            self.save(update_fields=['subscription_status'])
            
            if new_status == SiteSubscriptionStatus.EXPIRED:
                self.unpublish_site()

    def publish_site(self):
        """Publish site if subscription is active"""
        if self.subscription_status != SiteSubscriptionStatus.ACTIVE:
            raise ValidationError(_('Cannot publish without active subscription'))
        
        if self.status != SiteStatusChoices.APPROVED:
            raise ValidationError(_('Site must be approved before publishing'))

        self.status = SiteStatusChoices.PUBLISHED
        self.visibility = SiteVisibilityChoices.VISIBLE
        self.published_at = timezone.now()
        self.visibility_changed_at = timezone.now()
        self.save(update_fields=['status', 'visibility', 'published_at', 'visibility_changed_at'])

    def unpublish_site(self):
        """Unpublish site"""
        self.status = SiteStatusChoices.UNPUBLISHED
        self.visibility = SiteVisibilityChoices.HIDDEN
        self.unpublished_at = timezone.now()
        self.visibility_changed_at = timezone.now()
        self.save(update_fields=['status', 'visibility', 'unpublished_at', 'visibility_changed_at'])
    
    def validate_transition(self, new_status):
        valid_transitions = {
            SiteStatusChoices.DRAFT: [SiteStatusChoices.PENDING_REVIEW],
            SiteStatusChoices.PENDING_REVIEW: [SiteStatusChoices.APPROVED, SiteStatusChoices.REJECTED],
            SiteStatusChoices.APPROVED: [SiteStatusChoices.PUBLISHED],
            SiteStatusChoices.PUBLISHED: [SiteStatusChoices.UNPUBLISHED],
            SiteStatusChoices.REJECTED: [SiteStatusChoices.DRAFT],
            SiteStatusChoices.UNPUBLISHED: [SiteStatusChoices.PUBLISHED],
        }
        if new_status not in valid_transitions.get(self.status, []):
            raise StateTransitionError(
                f"Cannot transition from {self.status} to {new_status}"
            )

    def validate_promotion(self, promotion_data):
        """Validate promotion data before creation"""
        if not self.can_have_promotion():
            raise ValidationError(_('Site cannot have promotions'))
            
        if not self.price:
            raise ValidationError(_('Site must have a price to create promotions'))
            
        # Validate discount value
        discount_type = promotion_data.get('discount_type')
        discount_value = Decimal(str(promotion_data.get('discount_value')))
        
        if discount_type == 'percentage':
            if not (0 < discount_value <= 100):
                raise ValidationError(_('Percentage discount must be between 0 and 100'))
        else:  # fixed amount
            if discount_value >= self.price:
                raise ValidationError(_('Fixed discount cannot be greater than price'))
                
        # Check for overlapping promotions
        overlapping = self.site_promotions.filter(
            is_active=True,
            start_date__lte=promotion_data['end_date'],
            end_date__gte=promotion_data['start_date']
        ).exists()
        
        if overlapping:
            raise ValidationError(_('Cannot create overlapping promotions'))

    @property
    def latest_review(self):
        return self.reviews.order_by('-created_at').first()

    def can_submit_for_review(self):
        """Check if site can be submitted for review"""
        if self.status in [SiteStatusChoices.PENDING_REVIEW, SiteStatusChoices.APPROVED]:
            return False
        return self.has_required_fields()

    def has_required_fields(self):
        """Check if all required fields are filled"""
        required_fields = ['title', 'description', 'type', 'address']
        return all(getattr(self, field) for field in required_fields)

    def can_be_approved(self):
        """Check if site can be approved"""
        if not self.status == SiteStatusChoices.PENDING_REVIEW:
            return False
        if not self.latest_review:
            return False
        return self.latest_review.all_required_items_satisfied()

    def can_be_published(self):
        """Check if site can be published"""
        return (
            self.status == SiteStatusChoices.APPROVED and
            self.has_valid_subscription()
        )

    def has_valid_subscription(self):
        """Check if site has valid subscription"""
        if not self.active_subscription:
            return False
        return self.active_subscription.is_active()

    def clean(self):
        super().clean()
        if self.status == SiteStatusChoices.PUBLISHED:
            if not self.active_subscription:
                raise ValidationError(_('Site cannot be published without active subscription'))
            if not self.reviewed_by:
                raise ValidationError(_('Site must be reviewed before publishing'))

    def get_translated_field(self, field, language=None):
        if not language:
            return getattr(self, field)
        
        translation = self.translations.filter(language__code=language).first()
        if translation and hasattr(translation, field):
            return getattr(translation, field)
        return getattr(self, field)

    def average_rating(self):
        comments = self.comments.all()
        if comments.exists():
            return comments.aggregate(models.Avg('rating'))['rating__avg']
        return 0.0
    
    @property
    def is_visible(self):
        """
        Determines if site is publicly visible based on states
        """
        return (
            self.visibility_state == SiteVisibilityState.ACTIVE and
            self.subscription_state == SubscriptionState.ACTIVE
        )

    def can_transition_to(self, new_state):
        """
        Validates if state transition is allowed
        """
        from local_secrets.sites.constants import SITE_STATE_TRANSITIONS
        current_state = self.visibility_state
        allowed_transitions = SITE_STATE_TRANSITIONS.get(current_state, set())
        return new_state in allowed_transitions
 
    def generate_unique_slug(self):
        """Generate a unique slug based on title"""
        base_slug = slugify(self.title)
        slug = base_slug
        counter = 1
        
        while Site.objects.filter(slug=slug).exists():
            slug = f"{base_slug}-{counter}"
            counter += 1
            
        return slug
 
    def save(self, *args, **kwargs):
        # Generate slug if not provided
        if not self.slug:
            self.slug = self.generate_unique_slug()
        super().save(*args, **kwargs)
 
    class Meta:
        verbose_name = _('Site')
        verbose_name_plural = _('Sites')
        # constraints = [
        #     models.UniqueConstraint(
        #         fields=['title', 'type'],
        #         name='unique_site_title_type'
        #     ),
        #     models.CheckConstraint(
        #         check=models.Q(type__in=['place', 'event']),
        #         name='valid_site_type'
        #     )
        # ]
        indexes = [
            models.Index(fields=['frequency'], name='frequency_idx', condition=Q(type='event')),
            models.Index(fields=['status']),
            models.Index(fields=['subscription_status']),
            models.Index(fields=['created_at']),
            models.Index(fields=['visibility', 'status']),
            models.Index(fields=['visibility_state', 'subscription_state']),
            models.Index(fields=['state_changed_at']),
            models.Index(fields=['price']),
            models.Index(fields=['vendor', 'subscription_status', 'status']),
        ]

    def __str__(self):
        return self.title

class TranslatedSite(models.Model):
    site = models.ForeignKey(Site, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Translated Title'))
    description = models.TextField(verbose_name=_('Translated Description'))

    class Meta:
        indexes = [
            models.Index(fields=['title', 'description'])
        ]

class SiteStatusHistory(models.Model):
    site = models.ForeignKey('Site', on_delete=models.CASCADE, related_name='status_history')
    status = models.CharField(max_length=20, choices=SiteStatusChoices.choices)
    changed_by = models.ForeignKey('users.CustomUser', on_delete=models.SET_NULL, null=True)
    changed_at = models.DateTimeField(auto_now_add=True)
    feedback = models.TextField(null=True, blank=True)
    
    class Meta:
        ordering = ['-changed_at']
        indexes = [
            models.Index(fields=['site', 'status']),
            models.Index(fields=['changed_at']),
        ]
 
class SiteImage(models.Model):
    site = models.ForeignKey(Site, on_delete=models.CASCADE, related_name='images', verbose_name=_('Site'))
    image = ThumbnailerImageField(upload_to='site_images')
 
class Level(models.Model):
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Title'))
    type = models.CharField(max_length=100, choices=SiteType.choices, default=SiteType.PLACE, verbose_name=_('Type'))
    order = models.IntegerField(default=0, verbose_name=_('Order'))

    objects = BaseManager()
    objects_for_admin = BaseManager()
    objects_for_api = LevelManager()

    class Meta:
        verbose_name = _('Search level')
        verbose_name_plural = _('Search levels')
        ordering = ('title',)

    def display_text(self, field, language='en'):
        # Try fetching the specified translated field
        translation = self.translations.filter(language__code=language).first()
        if translation:
            return getattr(translation, field)
        # Fallback to the default language
        translation = self.translations.filter(language__code='es').first()
        if translation:
            return getattr(translation, field)
        # Fallback to the base model field
        return getattr(self, field)
 
class TranslatedLevel(models.Model):
    level = models.ForeignKey(Level, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Translated Title'))

    class Meta:
        indexes = [
            models.Index(fields=['title',])
        ]
 
class Category(models.Model):
    level = models.ForeignKey(
        Level, on_delete=models.SET_NULL, null=True, related_name='categories', verbose_name=_('Level')
    )
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Title'))
    type = models.CharField(max_length=100, choices=SiteType.choices, default=SiteType.PLACE, verbose_name=_('Type'))
    order = models.IntegerField(default=0, verbose_name=_('Order'))

    objects = BaseManager()
    objects_for_admin = BaseManager()
    objects_for_api = CategoryManager()

    class Meta:
        verbose_name = _('Category')
        verbose_name_plural = _('Categories')
        ordering = ('title',)

    def display_text(self, field, language='en'):
        # Try fetching the specified translated field
        translation = self.translations.filter(language__code=language).first()
        if translation:
            return getattr(translation, field)
        # Fallback to the default language
        translation = self.translations.filter(language__code='es').first()
        if translation:
            return getattr(translation, field)
        # Fallback to the base model field
        return getattr(self, field)

    def __str__(self):
        return f'{self.title} - ({self.type})'
 
class TranslatedCategory(models.Model):
    category = models.ForeignKey(Category, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Translated Title'))

    class Meta:
        indexes = [
            models.Index(fields=['title',])
        ]
 
class SubCategory(models.Model):
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Title'))
    type = models.CharField(max_length=100, choices=SiteType.choices, default=SiteType.PLACE, verbose_name=_('Type'))

    category = models.ForeignKey(
        Category, on_delete=models.SET_NULL, null=True, related_name='subcategories', verbose_name=_('Category')
    )
    order = models.IntegerField(default=0, verbose_name=_('Order'))

    objects = BaseManager()
    objects_for_admin = BaseManager()
    objects_for_api = SubcategoryManager()

    class Meta:
        verbose_name = _('Subcategory')
        verbose_name_plural = _('Subcategories')
        ordering = ('title',)

    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            return getattr(self, field)
        
 
 
class TranslatedSubCategory(models.Model):
    subcategory = models.ForeignKey(SubCategory, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    title = models.CharField(max_length=500, null=False, blank=False, verbose_name=_('Title'))

    class Meta:
        indexes = [
            models.Index(fields=['title',])
        ]
 
class FavoriteSites(models.Model):
    user = models.ForeignKey(get_user_model(), on_delete=models.CASCADE, verbose_name=_('User'))
    site = models.ForeignKey(Site, on_delete=models.CASCADE, verbose_name=_('Site'))

    class Meta:
        verbose_name = _('Favorite Site')
        verbose_name_plural = _('Favorite Sites')

    def __str__(self):
        return f'{self.user.username} likes {self.site.title}'


class Schedule(models.Model):
    day = models.CharField(max_length=100, choices=Day.choices)
    site = models.ForeignKey(Site, related_name='schedules', on_delete=models.CASCADE, verbose_name=_('Site'))

    objects = ScheduleQuerySet.as_manager()

    class Meta:
        verbose_name = _('Schedule')
        verbose_name_plural = _('Schedules')

    def __str__(self):
        try:
            return f'{self.site.title} - {self.day}'
        except Exception:
            return f'None - {self.day}'


class HourRange(models.Model):
    initial_hour = models.TimeField(default='08:00 AM')
    end_hour = models.TimeField(default='11:00 PM')
    schedule = models.ForeignKey(
        Schedule, related_name='opening_hours', on_delete=models.CASCADE, verbose_name=_('Schedule')
    )

    class Meta:
        verbose_name = _('Hour Range')
        verbose_name_plural = _('Hour Ranges')

    def __str__(self):
        return f'{self.schedule.day}: {self.initial_hour} - {self.end_hour}'


class SpecialSchedule(models.Model):
    day = models.DateField()
    site = models.ForeignKey(Site, related_name='special_schedules', on_delete=models.CASCADE, verbose_name=_('Site'))

    objects = SpecialScheduleQuerySet.as_manager()

    class Meta:
        verbose_name = _('Special Schedule')
        verbose_name_plural = _('Special Schedules')

    def __str__(self):
        return f'{self.day}'


class SpecialHourRange(models.Model):
    initial_hour = models.TimeField(default='08:00')
    end_hour = models.TimeField(default='23:00')
    schedule = models.ForeignKey(
        SpecialSchedule, related_name='opening_hours', on_delete=models.CASCADE, verbose_name=_('Schedule')
    )

    def __str__(self):
        return f'{self.schedule.day}: {self.initial_hour} - {self.end_hour}'


class Comment(models.Model):
    user = models.ForeignKey(
        get_user_model(), on_delete=models.CASCADE, related_name='comments', verbose_name=_('User')
    )
    site = models.ForeignKey(Site, on_delete=models.CASCADE, related_name='comments', verbose_name=_('Site'))
    body = models.TextField(verbose_name=_('Body'))
    #rating = models.IntegerField(verbose_name=_('Rating'), validators=[MaxValueValidator(5)])
    rating = models.DecimalField(
        max_digits=3, 
        decimal_places=1, 
        verbose_name=_('Rating'),
        validators=[MinValueValidator(0.0), MaxValueValidator(5.0)]
    )
    created_at = models.DateTimeField(verbose_name=_('Created at'))

    objects =  CommentQuerySet.as_manager()
    
    class Meta:
        verbose_name = _('Review')
        verbose_name_plural = _('Reviews')
    def __str__(self):
        return f'{self.user} - {self.site} - {self.rating}'


class DefaultImage(models.Model):
    title = models.CharField(max_length=100)
    image = ThumbnailerImageField(upload_to='default_images')

    def __str__(self):
        return self.title


class ImageSize(models.Model):
    min_width = models.IntegerField(default=512)
    min_height = models.IntegerField(default=512)
    max_width = models.IntegerField(default=4096)
    max_height = models.IntegerField(default=2160)

    def __str__(self):
        return 'ImageSize'

    class Meta:
        verbose_name = _('Image Size')
        verbose_name_plural = _('Images Sizes')


class VideoSize(models.Model):
    min_size = models.IntegerField(default=512)
    max_size = models.IntegerField(default=4096)

    def __str__(self):
        return 'VideoSize'

    class Meta:
        verbose_name = _('Video Size')
        verbose_name_plural = _('Videos Sizes')
  
class ReviewChecklistItem(models.Model):
    """Checklist items for site review"""
    name = models.CharField(max_length=100, verbose_name=_('Name'))
    description = models.TextField(verbose_name=_('Description'))
    is_required = models.BooleanField(default=True, verbose_name=_('Required'))
    order = models.PositiveIntegerField(default=0, verbose_name=_('Order'))
    
    class Meta:
        verbose_name = _('Review Checklist Item')
        verbose_name_plural = _('Review Checklist Items')
        ordering = ['order']

    def __str__(self):
        return self.name

class SiteReview(models.Model):
    """Detailed site review record"""
    site = models.ForeignKey(Site, on_delete=models.CASCADE, related_name='reviews', verbose_name=_('Site'))
    reviewer = models.ForeignKey(
        'users.CustomUser', 
        on_delete=models.SET_NULL, 
        null=True,
        related_name='site_reviews',
        verbose_name=_('Reviewer')
    )
    status = models.CharField(
        max_length=20,
        choices=SiteStatusChoices.choices,
        default=SiteStatusChoices.PENDING_REVIEW,
        verbose_name=_('Status')
    )
    feedback = models.TextField(blank=True, verbose_name=_('Feedback'))
    checklist_items = models.ManyToManyField(
        ReviewChecklistItem,
        through='ReviewChecklistResponse',
        verbose_name=_('Checklist Items')
    )
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_('Created At'))
    updated_at = models.DateTimeField(auto_now=True, verbose_name=_('Updated At'))
    
    class Meta:
        verbose_name = _('Site Review')
        verbose_name_plural = _('Site Reviews')
        ordering = ['-created_at']

    def __str__(self):
        return f"Review for {self.site.title}"

class ReviewChecklistResponse(models.Model):
    """Response to review checklist items"""
    review = models.ForeignKey(
        SiteReview, 
        on_delete=models.CASCADE, 
        related_name='checklist_responses',
        verbose_name=_('Review')
    )
    checklist_item = models.ForeignKey(
        ReviewChecklistItem, 
        on_delete=models.CASCADE,
        verbose_name=_('Checklist Item')
    )
    is_satisfied = models.BooleanField(default=False, verbose_name=_('Is Satisfied'))
    notes = models.TextField(blank=True, verbose_name=_('Notes'))
    
    class Meta:
        verbose_name = _('Review Checklist Response')
        verbose_name_plural = _('Review Checklist Responses')
        unique_together = ['review', 'checklist_item']

    def __str__(self):
        return f"{self.checklist_item.name} - {self.review.site.title}"

class SiteStateLog(models.Model):
    """
    Audit log for site state changes
    """
    site = models.ForeignKey(
        Site,
        on_delete=models.CASCADE,
        related_name='state_logs'
    )
    from_state = models.CharField(max_length=20)
    to_state = models.CharField(max_length=20)
    changed_at = models.DateTimeField(auto_now_add=True)
    changed_by = models.ForeignKey(
        settings.AUTH_USER_MODEL,  # Use AUTH_USER_MODEL setting
        on_delete=models.SET_NULL,
        null=True
    )
    reason = models.CharField(max_length=255, blank=True)
    metadata = models.JSONField(default=dict)

    class Meta:
        ordering = ['-changed_at']


class Promotion(models.Model):
    DISCOUNT_TYPE_CHOICES = [
        ('percentage', _('Percentage')),
        ('fixed', _('Fixed Amount'))
    ]
    
    # Fix the vendor reference
    vendor = models.ForeignKey(
        'users.Vendor',  # Correct reference to Vendor model in users app
        on_delete=models.CASCADE,
        related_name='site_promotions',
        verbose_name=_('Vendor')
    )
    
    site = models.ForeignKey(
        'sites.Site',
        on_delete=models.CASCADE,
        related_name='promotions',
        verbose_name=_('Site')
    )
    
    name = models.CharField(max_length=100, verbose_name=_('Name'))
    description = models.TextField(blank=True, verbose_name=_('Description'))
    discount_type = models.CharField(
        max_length=10,
        choices=DISCOUNT_TYPE_CHOICES,
        verbose_name=_('Discount Type')
    )
    discount_value = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        validators=[
            MinValueValidator(0),
            MaxValueValidator(100)
        ],
        verbose_name=_('Discount Value')
    )
    start_date = models.DateTimeField(verbose_name=_('Start Date'))
    end_date = models.DateTimeField(verbose_name=_('End Date'))
    is_active = models.BooleanField(default=True, verbose_name=_('Is Active'))
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)


    class Meta:
        verbose_name = _('Promotion')
        verbose_name_plural = _('Promotions')
        ordering = ['-created_at']
        indexes = [
            models.Index(fields=['vendor', 'site', 'is_active']),
            models.Index(fields=['start_date', 'end_date']),
        ]
        constraints = [
            models.CheckConstraint(
                check=models.Q(end_date__gt=models.F('start_date')),
                name='end_date_after_start_date'
            ),
            models.CheckConstraint(
                check=(
                    models.Q(
                        discount_type='percentage',
                        discount_value__gte=0,
                        discount_value__lte=100
                    ) |
                    models.Q(
                        discount_type='fixed',
                        discount_value__gte=0
                    )
                ),
                name='valid_discount_value'
            )
        ]


    def __str__(self):
        return f"{self.name} - {self.site.title}"


    def clean(self):
        super().clean()
        if self.start_date and self.end_date:
            if self.end_date <= self.start_date:
                raise ValidationError({
                    'end_date': _('End date must be after start date')
                })
        
        if self.start_date and self.start_date < timezone.now():
            raise ValidationError({
                'start_date': _('Start date cannot be in the past')
            })


        # Validate vendor ownership of site
        if self.site and self.vendor and self.site.vendor != self.vendor:
            raise ValidationError(_('Vendor can only create promotions for their own sites'))


    def save(self, *args, **kwargs):
        self.full_clean()
        super().save(*args, **kwargs)


class SiteViewLog(models.Model):
    site = models.ForeignKey(
        'Site',
        on_delete=models.CASCADE,
        related_name='view_logs'
    )
    user = models.ForeignKey(
        'users.CustomUser',
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='site_views'
    )
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    viewed_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        indexes = [
            models.Index(fields=['site', 'viewed_at']),
            models.Index(fields=['ip_address']),
            models.Index(fields=['user']),
        ]
        
    def __str__(self):
        return f"{self.site.title} viewed at {self.viewed_at}"