# local_secrets/sites/permissions.py

from rest_framework import permissions
from django.contrib.auth.mixins import UserPassesTestMixin
from rest_framework import permissions

class IsReviewerPermission(permissions.BasePermission):
    """
    Custom permission to only allow reviewers to access reviews.
    """
    def has_permission(self, request, view):
        return request.user.is_authenticated and (
            request.user.is_staff or 
            request.user.groups.filter(name='Ambassadors').exists()
        )

class ReviewerRequiredMixin(UserPassesTestMixin):
    """
    Mixin for Django admin views that requires the user to be a reviewer.
    """
    def test_func(self):
        return (
            self.request.user.is_authenticated and 
            (self.request.user.is_staff or 
             self.request.user.groups.filter(name='Ambassadors').exists())
        )

class IsVendorOrAdmin(permissions.BasePermission):
    """
    Custom permission to only allow vendors to access their own sites
    or admins to access all sites.
    """
    def has_permission(self, request, view):
        if request.user.is_staff:
            return True
            
        return hasattr(request.user, 'vendor_profile')

    def has_object_permission(self, request, view, obj):

        if request.user.is_staff:
            return True

        return obj.vendor.user == request.user

class HasActiveSubscription(permissions.BasePermission):
    """
    Custom permission to only allow vendors with active subscriptions.
    """
    def has_permission(self, request, view):
        if request.user.is_staff:
            return True
            
        if not hasattr(request.user, 'vendor_profile'):
            return False
            
        vendor = request.user.vendor_profile
        return vendor.has_active_subscription()

class CanSubmitForReview(permissions.BasePermission):
    """
    Custom permission to check if a site can be submitted for review.
    """
    def has_object_permission(self, request, view, obj):
        return obj.can_submit_for_review()
