# local_secrets/sites/services/state_management.py

from django.db import transaction
from django.utils import timezone
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _
from local_secrets.sites.models import Site, SiteStateLog, SiteVisibilityState, SubscriptionState
from local_secrets.sites.constants import SITE_STATE_TRANSITIONS
 
import logging

logger = logging.getLogger(__name__)

class StateTransitionError(ValidationError):
    pass

class SiteStateManager:
    """
    Manages site state transitions and enforces business rules
    """
    def __init__(self, site: Site):
        self.site = site
        self._original_state = site.visibility_state
        self._original_subscription_state = site.subscription_state

    @transaction.atomic
    def transition_to(self, new_state: str, user=None, reason=None, metadata=None):
        """
        Transitions site to new state with validation and logging
        """
        try:
            # Validate transition
            if not self.site.can_transition_to(new_state):
                raise StateTransitionError(
                    _("Invalid state transition from %(current)s to %(new)s") % {
                        'current': self.site.visibility_state,
                        'new': new_state
                    }
                )

            # Validate business rules
            self._validate_business_rules(new_state)

            # Store previous state for logging
            old_state = self.site.visibility_state

            # Update state
            self.site.visibility_state = new_state
            self.site.state_changed_at = timezone.now()
            self.site.state_changed_by = user
            self.site.state_change_reason = reason or ''
            self.site.save()

            # Create log entry
            self._create_state_log(
                from_state=old_state,
                to_state=new_state,
                user=user,
                reason=reason,
                metadata=metadata
            )

            # Trigger post-transition actions
            self._handle_post_transition_tasks(old_state, new_state)

            logger.info(
                "Site state transition successful",
                extra={
                    'site_id': self.site.id,
                    'from_state': old_state,
                    'to_state': new_state,
                    'user_id': user.id if user else None
                }
            )

            return True

        except Exception as e:
            logger.error(
                "Site state transition failed",
                extra={
                    'site_id': self.site.id,
                    'from_state': self.site.visibility_state,
                    'to_state': new_state,
                    'error': str(e)
                },
                exc_info=True
            )
            raise

    def _validate_business_rules(self, new_state: str):
        """
        Validates business rules for state transition
        """
        if new_state == SiteVisibilityState.ACTIVE:
            # Check subscription status
            if self.site.subscription_state != SubscriptionState.ACTIVE:
                raise StateTransitionError(
                    _("Cannot activate site without active subscription")
                )
            
            # Check quota limits
            if not self._check_quota_available():
                raise StateTransitionError(
                    _("Subscription quota exceeded")
                )

        elif new_state == SiteVisibilityState.PENDING_REVIEW:
            # Validate required fields
            if not self._validate_required_fields():
                raise StateTransitionError(
                    _("Required fields missing for review submission")
                )

    def _check_quota_available(self) -> bool:
        """
        Checks if vendor has available quota
        """
        from local_secrets.users.services import SubscriptionQuotaService
        
        quota_service = SubscriptionQuotaService(self.site.vendor)
        return quota_service.has_available_quota(self.site.type)

    def _validate_required_fields(self) -> bool:
        """
        Validates required fields for review submission
        """
        required_fields = ['title', 'description', 'address']
        return all(getattr(self.site, field) for field in required_fields)

    def _create_state_log(self, from_state: str, to_state: str, user=None, reason=None, metadata=None):
        """
        Creates state transition log entry
        """
        SiteStateLog.objects.create(
            site=self.site,
            from_state=from_state,
            to_state=to_state,
            changed_by=user,
            reason=reason or '',
            metadata=metadata or {}
        )

    def _handle_post_transition_tasks(self, old_state: str, new_state: str):
        """
        Handles post-transition tasks based on state change
        """
        if new_state == SiteVisibilityState.ACTIVE:
            self._handle_activation()
        elif new_state == SiteVisibilityState.INACTIVE:
            self._handle_deactivation()
        elif new_state == SiteVisibilityState.PENDING_REVIEW:
            self._handle_review_submission()

    def _handle_activation(self):
        """
        Handles tasks when site becomes active
        """
        # Update search index
        from local_secrets.searches.services import SearchIndexService
        SearchIndexService().index_site(self.site)

        # Send notifications
        self._send_activation_notifications()

    def _handle_deactivation(self):
        """
        Handles tasks when site becomes inactive
        """
        # Remove from search index
        from local_secrets.searches.services import SearchIndexService
        SearchIndexService().remove_site(self.site)

        # Send notifications
        self._send_deactivation_notifications()

    def _handle_review_submission(self):
        """
        Handles tasks when site is submitted for review
        """
        # Notify admins
        self._send_review_notifications()

    def _send_activation_notifications(self):
        """
        Sends notifications for site activation
        """
        from local_secrets.users.services import NotificationService
        
        notification_service = NotificationService()
        notification_service.notify_site_activation(self.site)
