# sites/signals.py
from django.db.models.signals import pre_save, post_save
from django.dispatch import receiver
from django.utils import timezone
from .models import Site, SiteViewLog
from local_secrets.sites.choices import SiteStatusChoices
from local_secrets.sites.notifications import SiteNotificationManager
from local_secrets.sites.state_machine import SiteStateMachine
from local_secrets.sites.validators import SiteValidator
from django.conf import settings
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _
from local_secrets.users.models import Notification, VendorSubscription
from django.contrib.auth.models import User
from django.db import models
from django.db.models import Q
from django.core.files.storage import default_storage
from django.core.files.base import ContentFile
from django.core.files import File
from django.core.files.images import get_image_dimensions
from django.core.files.uploadedfile import InMemoryUploadedFile
from django.core.files.uploadedfile import TemporaryUploadedFile

@receiver(pre_save, sender=Site)
def handle_status_change(sender, instance, **kwargs):
    """Handle status changes before save"""
    if not instance.pk:  
        return
        
    try:
        old_instance = Site.objects.get(pk=instance.pk)
        if old_instance.status != instance.status:
 
            SiteStateMachine.validate_transition(old_instance, instance.status)
 
            if instance.status == SiteStatusChoices.PUBLISHED:
                instance.published_at = timezone.now()
            elif instance.status == SiteStatusChoices.UNPUBLISHED:
                instance.unpublished_at = timezone.now()
                
    except Site.DoesNotExist:
        pass
 
@receiver(post_save, sender=Site)
def handle_status_notification(sender, instance, created, **kwargs):
    """Handle notifications after status change"""
    if not created and hasattr(instance, '_status_changed'):
        SiteNotificationManager.create_status_notification(
            site=instance,
            status=instance.status,
            user=instance.vendor.user if instance.vendor else None
        )

@receiver(post_save, sender=Site)
def handle_subscription_check(sender, instance, **kwargs):
    """Handle subscription checks and auto-unpublish"""
    if instance.status == SiteStatusChoices.PUBLISHED:
        if not instance.active_subscription or \
           (instance.active_subscription.end_date and \
            instance.active_subscription.end_date < timezone.now().date()):
            instance.unpublish_site()
            instance.save()


@receiver(pre_save, sender=Site)
def prevent_type_change(sender, instance, **kwargs):
    """
    Prevent changing the type of an existing site
    """
    if instance.pk:   
        try:
            original = sender.objects.get(pk=instance.pk)
            if original.type != instance.type:
                raise ValidationError(
                    "Cannot change the type of an existing site"
                )
        except sender.DoesNotExist:
            pass


@receiver(pre_save, sender=VendorSubscription)
def handle_subscription_status_change(sender, instance, **kwargs):
    if instance.pk:  # Only for existing subscriptions
        old_instance = VendorSubscription.objects.get(pk=instance.pk)
        
        # If subscription becomes inactive or expires
        if (old_instance.status == 'active' and instance.status != 'active') or \
           (instance.end_date and instance.end_date < timezone.now()):
            
            # Update all associated sites
            for site_subscription in instance.site_subscriptions.filter(is_active=True):
                site = site_subscription.site
                site.subscription_status = 'expired'
                site.is_published = False
                site.save()
                
                site_subscription.is_active = False
                site_subscription.save()

 
@receiver(post_save, sender=Site)
def update_site_analytics(sender, instance, **kwargs):
    """
    Signal to update analytics when site view count changes
    """
    if not kwargs.get('created', False):  # Only for updates
        try:
            if instance.tracker.has_changed('view_count'):
                old_count = instance.tracker.previous('view_count') or 0
                new_count = instance.view_count
                difference = new_count - old_count
                
                if difference > 0:
                    # Create view logs for the difference
                    SiteViewLog.objects.create(
                        site=instance,
                        viewed_at=timezone.now(),
                    )
                    
                    # Clear related caches
                    from django.core.cache import cache
                    cache.delete(f'site_views_{instance.id}_day')
                    cache.delete(f'site_views_{instance.id}_week')
                    cache.delete(f'site_views_{instance.id}_month')
                    cache.delete(f'site_views_realtime_{instance.id}')
        except AttributeError:
            pass 
