# sites/tests/test_publication.py

import pytest
from django.test import TestCase
from django.core.exceptions import ValidationError
from django.utils import timezone
from ..models import Site, SiteReview
from ..services import ReviewWorkflowManager
from .factories import (
    SiteFactory,
    VendorFactory,
    SubscriptionFactory,
    UserFactory
)

@pytest.mark.django_db
class TestSitePublication(TestCase):
    def setUp(self):
        self.vendor = VendorFactory()
        self.subscription = SubscriptionFactory(vendor=self.vendor)
        self.site = SiteFactory(
            vendor=self.vendor,
            active_subscription=self.subscription
        )
        self.admin = UserFactory(is_staff=True)
        
    def test_submit_for_review_workflow(self):
        workflow = ReviewWorkflowManager(self.site)
        review = workflow.submit_for_review()
        
        self.assertEqual(self.site.status, SiteStatusChoices.PENDING_REVIEW)
        self.assertIsNotNone(self.site.submitted_for_review_at)
        self.assertEqual(review.status, SiteStatusChoices.PENDING_REVIEW)
        
    def test_approve_review_workflow(self):
        workflow = ReviewWorkflowManager(self.site)
        workflow.submit_for_review()
        
        # Satisfy all checklist items
        for response in self.site.latest_review.checklist_responses.all():
            response.is_satisfied = True
            response.save()
            
        workflow.approve_review(self.admin, "Approved")
        
        self.assertEqual(self.site.status, SiteStatusChoices.APPROVED)
        self.assertIsNotNone(self.site.approved_at)
        
    def test_subscription_validation(self):
        # Expire subscription
        self.subscription.end_date = timezone.now().date()
        self.subscription.save()
        
        with self.assertRaises(ValidationError):
            SitePublicationValidator.validate_site_subscription(self.site)
            
    def test_publication_requirements(self):
        workflow = ReviewWorkflowManager(self.site)
        review = workflow.submit_for_review()
        
        # Try to publish without approval
        with self.assertRaises(ValidationError):
            SitePublicationValidator.validate_publication_requirements(self.site)
            
        # Approve and try again
        workflow.approve_review(self.admin, "Approved")
        SitePublicationValidator.validate_publication_requirements(self.site)
