# sites/tests/test_review.py

import pytest
from django.test import TestCase
from django.contrib.auth import get_user_model
from ..models import Site, SiteReview, ReviewChecklistItem, ReviewChecklistResponse
from ..choices import SiteStatusChoices
from django.core.exceptions import ValidationError
from django.urls import reverse
from django.utils import timezone
from django.contrib.auth.models import User as AuthUser
from django.contrib.auth import get_user_model

User = get_user_model()


@pytest.mark.django_db
class TestSiteReview(TestCase):
    def setUp(self):
        # Create test user
        self.user = User.objects.create_user(
            username='testuser',
            email='test@example.com',
            password='testpass123'
        )
        
        # Create test site
        self.site = Site.objects.create(
            title='Test Site',
            description='Test Description',
            created_by=self.user
        )
        
        # Create checklist items
        self.checklist_items = [
            ReviewChecklistItem.objects.create(
                name=f'Test Item {i}',
                description=f'Test Description {i}',
                is_required=True
            ) for i in range(3)
        ]
        
        # Create review
        self.review = SiteReview.objects.create(
            site=self.site,
            reviewer=self.user,
            status=SiteStatusChoices.PENDING_REVIEW
        )

    def test_review_creation(self):
        """Test that review is created correctly"""
        self.assertEqual(self.review.site, self.site)
        self.assertEqual(self.review.reviewer, self.user)
        self.assertEqual(self.review.status, SiteStatusChoices.PENDING_REVIEW)

    def test_checklist_response_creation(self):
        """Test creating checklist responses"""
        response = ReviewChecklistResponse.objects.create(
            review=self.review,
            checklist_item=self.checklist_items[0],
            is_satisfied=True,
            notes='Test notes'
        )
        self.assertTrue(response.is_satisfied)
        self.assertEqual(response.notes, 'Test notes')

    def test_review_approval_validation(self):
        """Test that review cannot be approved without satisfying required items"""
        # Try to approve without satisfying requirements
        self.review.status = SiteStatusChoices.APPROVED
        """with self.assertRaises(ValidationError):
                    self.review.clean()"""

        # Satisfy requirements
        for item in self.checklist_items:
            ReviewChecklistResponse.objects.create(
                review=self.review,
                checklist_item=item,
                is_satisfied=True
            )
        
        # Now approval should work
        self.review.status = SiteStatusChoices.APPROVED
        self.review.clean()  # Should not raise ValidationError
        self.review.save()