# local_secrets/sites/tests/test_visibility.py
from django.conf import settings
from django.test import TestCase, Client
from django.urls import reverse
from django.utils import timezone
from django.contrib.auth import get_user_model
from datetime import timedelta
from local_secrets.sites.models import Site, SiteVisibilityChoices
from local_secrets.sites.services.site_visibility import VisibilityService
from local_secrets.users.models import Vendor, VendorSubscription
from local_secrets.payment.models import VendorPayment
from local_secrets.core.events import SystemEvent 
from local_secrets.sites.cron import VisibilityCronJob


class VisibilityServiceTests(TestCase):
    def setUp(self):
        self.vendor = Vendor.objects.create(name="Test Vendor")
        self.site = Site.objects.create(
        vendor=self.vendor,
        visibility=SiteVisibilityChoices.VISIBLE
        )
        self.service = VisibilityService()
    def test_update_site_visibility(self):
        result = self.service.update_site_visibility(
            self.site,
            SiteVisibilityChoices.PAYMENT_PENDING,
            "Test update"
        )
        
        self.assertTrue(result)
        self.site.refresh_from_db()
        self.assertEqual(
            self.site.visibility,
            SiteVisibilityChoices.PAYMENT_PENDING
        )
        
        # Check event was created
        event = SystemEvent.objects.filter(
            event_type='site.visibility.changed'
        ).first()
        self.assertIsNotNone(event)
        self.assertEqual(event.payload['site_id'], self.site.id)


    def test_visibility_update_failure(self):
        # Simulate database error
        with self.assertLogs('local_secrets.sites', level='ERROR'):
            result = self.service.update_site_visibility(
                self.site,
                'invalid_status',
                "Test failure"
            )
            
        self.assertFalse(result)
        self.assertTrue(self.site.visibility_update_pending)
        self.assertEqual(self.site.update_attempts, 1)


class VisibilityCronJobTests(TestCase):
    def setUp(self):
        self.vendor = Vendor.objects.create(name="Test Vendor")
        self.site = Site.objects.create(
        vendor=self.vendor,
        visibility=SiteVisibilityChoices.VISIBLE,
        visibility_update_pending=True,
        update_attempts=1
        )
        self.cron = VisibilityCronJob()
    def test_process_pending_updates(self):
        self.cron.process_pending_visibility_updates()
        
        self.site.refresh_from_db()
        self.assertFalse(self.site.visibility_update_pending)
        
        # Check event was created
        event = SystemEvent.objects.filter(
            event_type='site.visibility.changed'
        ).first()
        self.assertIsNotNone(event)


    def test_max_retries_reached(self):
        self.site.update_attempts = settings.VISIBILITY_MAX_RETRIES
        self.site.save()
        
        self.cron.process_pending_visibility_updates()
        
        self.site.refresh_from_db()
        self.assertTrue(self.site.visibility_update_pending)
        
        # Check failure event was created
        event = SystemEvent.objects.filter(
            event_type='site.visibility.update_failed'
        ).first()
        self.assertIsNotNone(event)


class VisibilityAdminTests(TestCase):
    def setUp(self):
        User = get_user_model()
        self.admin_user = User.objects.create_superuser(
        'admin@test.com',
        'password123'
        )
        self.client = Client()
        self.client.force_login(self.admin_user)
        self.vendor = Vendor.objects.create(name="Test Vendor")
        self.site = Site.objects.create(
            vendor=self.vendor,
            visibility=SiteVisibilityChoices.VISIBLE
        )


    def test_view_dashboard(self):
        response = self.client.get(reverse('admin:visibility_dashboard'))
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(
            response,
            'sites/dashboard/visibility_dashboard.html'
        )


    def test_reset_visibility(self):
        self.site.visibility_update_pending = True
        self.site.update_attempts = 2
        self.site.save()
        
        response = self.client.post(
            reverse('admin:reset-visibility', args=[self.site.id])
        )
        
        self.site.refresh_from_db()
        self.assertFalse(self.site.visibility_update_pending)
        self.assertEqual(self.site.update_attempts, 0)


