# local_secrets/sites/utils.py
from django.conf import settings
from django.contrib.gis.geos import Point
from django.contrib.gis.db.models.functions import Distance
from django.contrib.gis.measure import D
from django.core.cache import cache
from django.conf import settings
from rest_framework.throttling import AnonRateThrottle, UserRateThrottle
from rest_framework.exceptions import Throttled
import hashlib
import json
from datetime import timedelta
from django_redis.cache import RedisCache
import logging

logger = logging.getLogger(__name__)

class SearchRateThrottle(UserRateThrottle):
    rate = '30/minute'  # Authenticated users
    
class AnonSearchRateThrottle(AnonRateThrottle):
    rate = '10/minute'  # Anonymous users

class CacheManager:
    def __init__(self):
        self.cache = cache  # Use Django's default cache backend
        self.cache_timeout = getattr(settings, 'SITE_SEARCH_CACHE_TIMEOUT', 3600)
        self.cache_enabled = getattr(settings, 'SITE_SEARCH_CACHE_ENABLED', True)
        
    def generate_cache_key(self, query, params, user_id=None):
        """Generate unique cache key based on search parameters"""
        cache_dict = {
            'query': query,
            'params': params,
            'user_id': user_id
        }
        cache_string = json.dumps(cache_dict, sort_keys=True)
        return f"site_search_{hashlib.md5(cache_string.encode()).hexdigest()}"
    
    def get_cached_results(self, key):
        """Get results from cache"""
        if not self.cache_enabled:
            return None, None
            
        try:
            results = self.cache.get(key)
            return results, 'cache' if results else None
        except Exception as e:
            logger.error(f"Cache retrieval error: {str(e)}")
            return None, None
        
    def cache_results(self, key, results, timeout=None):
        """Cache results with fallback timeout"""
        if not self.cache_enabled:
            return
            
        try:
            self.cache.set(
                key, 
                results, 
                timeout=timeout or self.cache_timeout
            )
        except Exception as e:
            logger.error(f"Cache storage error: {str(e)}")


def get_location_message(message_key, **kwargs):
    """
    Get customized location message
    """
    message = settings.LOCATION_MESSAGES.get(message_key, '')
    if kwargs:
        message = message.format(**kwargs)
    return message