# sites/views/admin.py

from django.views.generic import ListView, DetailView, UpdateView
from django.contrib.auth.mixins import UserPassesTestMixin
from django.urls import reverse_lazy
from django.shortcuts import redirect
from django.contrib import messages
from django.utils.translation import gettext as _
from django.core.exceptions import ValidationError

from local_secrets.sites.choices import SiteStatusChoices
from local_secrets.sites.models import ReviewChecklistItem, Site
from local_secrets.sites.serializers import SiteReviewActionSerializer
from local_secrets.sites.services.review_workflow import ReviewWorkflowManager

class AdminReviewListView(UserPassesTestMixin, ListView):
    template_name = 'sites/admin/review_list.html'
    context_object_name = 'pending_reviews'
    
    def test_func(self):
        return self.request.user.is_staff
    
    def get_queryset(self):
        return Site.objects.filter(
            status=SiteStatusChoices.PENDING_REVIEW
        ).select_related(
            'vendor',
            'latest_review'
        ).order_by('submitted_for_review_at')

class AdminReviewDetailView(UserPassesTestMixin, DetailView):
    template_name = 'sites/admin/review_detail.html'
    context_object_name = 'site'
    
    def test_func(self):
        return self.request.user.is_staff
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        site = self.get_object()
        context.update({
            'checklist_items': ReviewChecklistItem.objects.all(),
            'review_form': SiteReviewActionSerializer(instance=site.latest_review),
            'subscription_status': site.has_valid_subscription()
        })
        return context

class AdminReviewActionView(UserPassesTestMixin, UpdateView):
    template_name = 'sites/admin/review_action.html'
    form_class = SiteReviewActionSerializer
    
    def test_func(self):
        return self.request.user.is_staff
    
    def form_valid(self, form):
        site = self.get_object()
        workflow = ReviewWorkflowManager(site)
        
        action = form.cleaned_data['action']
        feedback = form.cleaned_data['feedback']
        
        try:
            if action == 'approve':
                workflow.approve_review(self.request.user, feedback)
                messages.success(self.request, _("Site has been approved"))
            elif action == 'reject':
                workflow.reject_review(self.request.user, feedback)
                messages.warning(self.request, _("Site has been rejected"))
                
            return redirect('admin:review-list')
        except ValidationError as e:
            messages.error(self.request, str(e))
            return self.form_invalid(form)
