# utils.py  (or wherever you added it)
from django.db import models
from local_secrets.sites.models import Site
from local_secrets.travels.models import Stop, Travel
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

def add_stop_safely(travel: Travel, site_id: int, order: int | None = None) -> Stop:
    """
    Create (or update) a Stop linking *travel* and *site_id*.
    Guarantees:
    • one Stop per (travel, site)
    • a valid sequential order
    • city auto-linked to the travel
    """
    site = Site.objects.get(pk=site_id)

    if order is None:                                       # default next order
        last = Stop.objects.filter(travel=travel).aggregate(
            max=models.Max("order")
        )["max"] or 0
        order = last + 1

    stop, created = Stop.objects.get_or_create(
        travel=travel,
        site=site,
        defaults={"order": order},
    )
    if not created and stop.order != order:
        stop.order = order
        stop.save(update_fields=["order"])

    # keep Travel.cities in sync
    if site.city_id and not travel.cities.filter(pk=site.city_id).exists():
        travel.cities.add(site.city_id)

    logging.getLogger(__name__).info(
        "Stop %s for travel %s (site %s, order %s)",
        "created" if created else "updated",
        travel.id,
        site.id,
        order,
    )
    return stop