# local_secrets/users/emails.py

from django.conf import settings
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.utils.translation import gettext_lazy as _
from django.urls import reverse

class VendorEmailNotifications:
    """Handle all vendor-related email notifications"""
    
    @staticmethod
    def send_subscription_created(subscription):
        """Send email when a new subscription is created"""
        context = {
            'vendor_name': subscription.vendor.company_name,
            'plan_name': subscription.plan.name,
            'start_date': subscription.start_date,
            'end_date': subscription.end_date,
            'amount': subscription.plan.price,
            'dashboard_url': settings.FRONTEND_URL + reverse('vendor-dashboard')
        }
        
        send_mail(
            subject=_('Your subscription has been activated'),
            message=render_to_string('users/emails/subscription_created.txt', context),
            html_message=render_to_string('users/emails/subscription_created.html', context),
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[subscription.vendor.user.email],
            fail_silently=False
        )

    @staticmethod
    def send_subscription_expiring_soon(subscription):
        """Send email when subscription is about to expire"""
        context = {
            'vendor_name': subscription.vendor.company_name,
            'plan_name': subscription.plan.name,
            'end_date': subscription.end_date,
            'renewal_url': settings.FRONTEND_URL + reverse('subscription-renewal')
        }
        
        send_mail(
            subject=_('Your subscription is expiring soon'),
            message=render_to_string('users/emails/subscription_expiring.txt', context),
            html_message=render_to_string('users/emails/subscription_expiring.html', context),
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[subscription.vendor.user.email],
            fail_silently=False
        )

    @staticmethod
    def send_site_claim_status(claim):
        """Send email when a site claim is processed"""
        context = {
            'vendor_name': claim.vendor.company_name,
            'site_name': claim.site.title,
            'status': claim.status,
            'notes': claim.notes,
            'dashboard_url': settings.FRONTEND_URL + reverse('vendor-dashboard')
        }
        
        send_mail(
            subject=_('Your site claim has been processed'),
            message=render_to_string('users/emails/site_claim_status.txt', context),
            html_message=render_to_string('users/emails/site_claim_status.html', context),
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[claim.vendor.user.email],
            fail_silently=False
        )
