# local_secrets/users/forms.py

from django import forms
from django.utils.translation import gettext_lazy as _
from .models import SiteClaim, Vendor, BusinessType
from ..sites.models import Site
from .models import Vendor, BusinessType
from django.utils.text import slugify 

class SiteClaimForm(forms.ModelForm):
    class Meta:
        model = SiteClaim
        fields = '__all__'
        widgets = {
            'notes': forms.Textarea(attrs={'rows': 4}),
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['site'].queryset = Site.objects.select_related('city', 'vendor')
        self.fields['vendor'].queryset = Vendor.objects.select_related('user')

    def clean(self):
        cleaned_data = super().clean()
        site = cleaned_data.get('site')
        vendor = cleaned_data.get('vendor')

        if site and vendor:
            if site.vendor and site.vendor != vendor:
                raise forms.ValidationError(
                    _("This site is already claimed by another vendor.")
                )
            
            existing_claims = SiteClaim.objects.filter(
                site=site,
                vendor=vendor,
                status='PENDING'
            ).exclude(pk=self.instance.pk if self.instance else None)
            
            if existing_claims.exists():
                raise forms.ValidationError(
                    _("There is already a pending claim for this site by this vendor.")
                )

        return cleaned_data
 
class VendorRegistrationForm(forms.ModelForm):
    class Meta:
        model = Vendor
        fields = [
            'company_name',
            'contact_email',
            'description',
            'business_type',
            'contact_phone',
            'contact_email',
            'website',
            'address'
        ]
        widgets = {
            'description': forms.Textarea(attrs={'rows': 4}),
            'address': forms.Textarea(attrs={'rows': 3}),
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['business_type'].queryset = BusinessType.objects.all()
        
        # Add CSS classes and placeholders
        for field in self.fields:
            self.fields[field].widget.attrs.update({
                'class': 'form-control',
                'placeholder': self.fields[field].label
            })

    def clean_company_name(self):
        company_name = self.cleaned_data['company_name']
        slug = slugify(company_name)
        
        # Check if a vendor with this slug already exists
        if Vendor.objects.filter(slug=slug).exists():
            raise forms.ValidationError(
                'A vendor with a similar name already exists. Please choose a different name.'
            )
        return company_name

    def save(self, commit=True):
        instance = super().save(commit=False)
        instance.slug = slugify(instance.company_name)
        if commit:
            instance.save()
        return instance
 
class VendorProfileForm(forms.ModelForm):
    class Meta:
        model = Vendor
        fields = [
            'company_name',
            'contact_email',
            'contact_phone',
            'contact_email',
            'website',
            'description',
            'address',
            'business_type'
        ]
        widgets = {
            'description': forms.Textarea(attrs={'rows': 4}),
            'address': forms.Textarea(attrs={'rows': 3}),
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for field in self.fields:
            self.fields[field].widget.attrs.update({'class': 'form-control'})
