# local_secrets/users/permissions.py

from rest_framework import permissions
from django.utils.translation import gettext_lazy as _

class IsVendorOwner(permissions.BasePermission):
    """
    Custom permission to only allow owners of a vendor to access it.
    """
    def has_object_permission(self, request, view, obj):
        return obj.vendor.user == request.user

class HasActiveSubscription(permissions.BasePermission):
    """
    Custom permission to only allow vendors with active subscriptions.
    """
    message = _("You need an active subscription to perform this action.")

    def has_permission(self, request, view):
        if not request.user.is_authenticated:
            return False
            
        if hasattr(request.user, 'vendor_profile'):
            subscription = request.user.vendor_profile.subscriptions.filter(
                status='active'
            ).first()
            return subscription is not None
        return False

class HasAvailableQuota(permissions.BasePermission):
    """
    Custom permission to check if vendor has available quota.
    """
    message = _("You have reached your quota limit.")

    def has_permission(self, request, view):
        if not request.user.is_authenticated:
            return False
            
        if hasattr(request.user, 'vendor_profile'):
            from local_secrets.users.services.subscription_services import SubscriptionService
            
            site_type = request.data.get('type', 'site')
            return SubscriptionService.check_quota_available(
                request.user.vendor_profile,
                site_type
            )
        return False

class IsAuthenticated(permissions.BasePermission):
    """
    Custom permission to only allow authenticated users.
    """
    message = _("You must be logged in to perform this action.")

    def has_permission(self, request, view):
        return request.user.is_authenticated