# local_secrets/users/services/bulk_operations.py

from datetime import timedelta
from django.db import transaction
from django.core.exceptions import ValidationError
from django.utils import timezone
from django.utils.translation import gettext_lazy as _
from local_secrets.users.emails import VendorEmailNotifications
from local_secrets.users.models import Vendor, VendorSubscription, SiteClaim
from local_secrets.sites.models import Site
from local_secrets.users.services.subscription_services import SubscriptionService 


class BulkOperationsService:
    @staticmethod
    @transaction.atomic
    def process_claims(claim_ids, action, admin_user, notes=None):
        """
        Process multiple site claims at once
        """
        if action not in ['approve', 'reject']:
            raise ValidationError("Invalid action specified")

        claims = SiteClaim.objects.filter(
            id__in=claim_ids,
            status='PENDING'
        ).select_related('site', 'vendor')

        results = {
            'processed': 0,
            'failed': 0,
            'errors': []
        }

        for claim in claims:
            try:
                status = 'APPROVED' if action == 'approve' else 'REJECTED'
                
                # Process claim
                claim.status = status
                claim.processed_at = timezone.now()
                claim.processed_by = admin_user
                claim.notes = notes or ''
                claim.save()

                if status == 'APPROVED':
                    # Update site ownership
                    claim.site.vendor = claim.vendor
                    claim.site.save()

                    # Reject other pending claims
                    SiteClaim.objects.filter(
                        site=claim.site,
                        status='PENDING'
                    ).exclude(id=claim.id).update(
                        status='REJECTED',
                        processed_at=timezone.now(),
                        processed_by=admin_user,
                        notes="Site claimed by another vendor"
                    )

                results['processed'] += 1
            except Exception as e:
                results['failed'] += 1
                results['errors'].append(f"Claim {claim.id}: {str(e)}")

        return results

    @staticmethod
    @transaction.atomic
    def update_subscriptions(subscription_ids, action, **kwargs):
        """
        Update multiple subscriptions at once
        """
        valid_actions = ['cancel', 'renew', 'change_plan']
        if action not in valid_actions:
            raise ValidationError("Invalid action specified")

        subscriptions = VendorSubscription.objects.filter(
            id__in=subscription_ids
        ).select_related('vendor', 'plan')

        results = {
            'processed': 0,
            'failed': 0,
            'errors': []
        }

        for subscription in subscriptions:
            try:
                if action == 'cancel':
                    subscription.status = 'cancelled'
                    subscription.cancelled_at = timezone.now()
                    subscription.save()
                
                elif action == 'renew':
                    SubscriptionService.process_renewal(subscription)
                
                elif action == 'change_plan':
                    new_plan = kwargs.get('new_plan')
                    if not new_plan:
                        raise ValidationError("New plan not specified")
                    
                    subscription.plan = new_plan
                    subscription.save()

                results['processed'] += 1
            except Exception as e:
                results['failed'] += 1
                results['errors'].append(f"Subscription {subscription.id}: {str(e)}")

        return results

    @staticmethod
    @transaction.atomic
    def bulk_site_operations(site_ids, action, **kwargs):
        """
        Process multiple site operations
        """
        valid_actions = ['publish', 'unpublish', 'archive', 'delete', 'change_status']
        if action not in valid_actions:
            raise ValidationError("Invalid action specified")

        sites = Site.objects.filter(id__in=site_ids)
        user = kwargs.get('user')
        
        results = {
            'processed': 0,
            'failed': 0,
            'errors': [],
            'sites': []
        }

        for site in sites:
            try:
                if action == 'publish':
                    if not site.vendor.has_active_subscription():
                        raise ValidationError("Active subscription required")
                    site.status = 'published'
                    site.published_at = timezone.now()
                
                elif action == 'unpublish':
                    site.status = 'unpublished'
                    site.unpublished_at = timezone.now()
                
                elif action == 'archive':
                    site.status = 'archived'
                    site.archived_at = timezone.now()
                
                elif action == 'delete':
                    site.delete()
                    results['sites'].append(f"Site {site.id} deleted")
                    continue
                
                elif action == 'change_status':
                    new_status = kwargs.get('new_status')
                    if not new_status:
                        raise ValidationError("New status not specified")
                    site.status = new_status

                site.save()
                results['processed'] += 1
                results['sites'].append(f"Site {site.id} {action}ed")
                
            except Exception as e:
                results['failed'] += 1
                results['errors'].append(f"Site {site.id}: {str(e)}")

        return results

    @staticmethod
    @transaction.atomic
    def bulk_vendor_operations(vendor_ids, action, **kwargs):
        """
        Process multiple vendor operations
        """
        valid_actions = ['activate', 'deactivate', 'update_subscription', 'send_notification']
        if action not in valid_actions:
            raise ValidationError("Invalid action specified")

        vendors = Vendor.objects.filter(id__in=vendor_ids)
        
        results = {
            'processed': 0,
            'failed': 0,
            'errors': []
        }

        for vendor in vendors:
            try:
                if action == 'activate':
                    vendor.is_active = True
                    vendor.activated_at = timezone.now()
                
                elif action == 'deactivate':
                    vendor.is_active = False
                    vendor.deactivated_at = timezone.now()
                
                elif action == 'update_subscription':
                    plan = kwargs.get('plan')
                    if not plan:
                        raise ValidationError("Plan not specified")
                    
                    subscription = vendor.get_active_subscription()
                    if subscription:
                        subscription.plan = plan
                        subscription.save()
                
                elif action == 'send_notification':
                    notification_type = kwargs.get('notification_type')
                    message = kwargs.get('message')
                    VendorEmailNotifications.send_custom_notification(
                        vendor, notification_type, message
                    )

                vendor.save()
                results['processed'] += 1
                
            except Exception as e:
                results['failed'] += 1
                results['errors'].append(f"Vendor {vendor.id}: {str(e)}")

        return results

    @staticmethod
    @transaction.atomic
    def bulk_subscription_updates(subscription_ids, action, **kwargs):
        """
        Process multiple subscription updates
        """
        valid_actions = ['extend', 'apply_discount', 'change_auto_renew', 'update_quotas']
        if action not in valid_actions:
            raise ValidationError("Invalid action specified")

        subscriptions = VendorSubscription.objects.filter(id__in=subscription_ids)
        
        results = {
            'processed': 0,
            'failed': 0,
            'errors': []
        }

        for subscription in subscriptions:
            try:
                if action == 'extend':
                    days = kwargs.get('days', 0)
                    subscription.end_date = subscription.end_date + timedelta(days=days)
                
                elif action == 'apply_discount':
                    discount = kwargs.get('discount', 0)
                    subscription.apply_discount(discount)
                
                elif action == 'change_auto_renew':
                    auto_renew = kwargs.get('auto_renew', True)
                    subscription.auto_renew = auto_renew
                
                elif action == 'update_quotas':
                    new_quotas = kwargs.get('quotas', {})
                    subscription.update_quotas(new_quotas)

                subscription.save()
                results['processed'] += 1
                
            except Exception as e:
                results['failed'] += 1
                results['errors'].append(f"Subscription {subscription.id}: {str(e)}")

        return results