# local_secrets/users/services/payment_services.py
from datetime import timezone
from decimal import Decimal
from typing import Any, Dict, Optional
import stripe
from django.conf import settings
from django.urls import reverse
from django.utils.translation import gettext_lazy as _
import logging
from local_secrets.payment.exceptions import PaymentProcessError
from local_secrets.payment.models import VendorPayment
from local_secrets.payment.processors import PayPalPaymentProcessor, StripePaymentProcessor
from local_secrets.sites.services.state_management import SiteStateManager
from local_secrets.users.models import SubscriptionPlan, Vendor, VendorSubscription
from django.db import transaction
logger = logging.getLogger(__name__)

stripe.api_key = settings.STRIPE_SECRET_KEY


class PaymentService:
    """
    High-level service for payment operations that coordinates between
    payment processors and business logic
    """
    PROCESSOR_MAPPING = {
        'stripe': StripePaymentProcessor,
        'paypal': PayPalPaymentProcessor
    }

    def __init__(self, payment: VendorPayment):
        self.payment = payment
        self.processor = self._get_processor()

    def _get_processor(self):
        """
        Gets appropriate payment processor based on payment method
        """
        processor_class = self.PROCESSOR_MAPPING.get(self.payment.payment_method.provider)
        if not processor_class:
            raise PaymentProcessError(f"Unsupported payment provider: {self.payment.payment_method.provider}")
        
        gateway = self._get_gateway(self.payment.payment_method.provider)
        return processor_class(self.payment, gateway)

    def _get_gateway(self, provider: str):
        """
        Gets payment gateway interface based on provider
        """
        from local_secrets.payment.gateways.stripe import StripeGateway
        from local_secrets.payment.gateways.paypal import PayPalGateway

        gateways = {
            'stripe': StripeGateway,
            'paypal': PayPalGateway
        }

        gateway_class = gateways.get(provider)
        if not gateway_class:
            raise PaymentProcessError(f"No gateway configured for provider: {provider}")
        
        return gateway_class()

    @transaction.atomic
    def process_payment(self) -> Dict[str, Any]:
        """
        Processes payment and handles business logic
        """
        try:
            # Process payment through processor
            response = self.processor.process_payment()
            
            # Handle post-payment business logic if payment was successful
            if self.payment.status == 'completed':
                self._handle_successful_payment(response)
            
            return response

        except Exception as e:
            logger.error(
                "Payment processing failed",
                extra={
                    'payment_id': self.payment.id,
                    'vendor_id': self.payment.vendor.id,
                    'error': str(e)
                },
                exc_info=True
            )
            raise

    @transaction.atomic
    def process_refund(self, amount: Optional[Decimal] = None, reason: str = None) -> Dict[str, Any]:
        """
        Processes refund and handles business logic
        """
        try:
            # Process refund through processor
            response = self.processor.refund(amount)
            
            # Handle post-refund business logic
            if response.get('status') == 'refunded':
                self._handle_successful_refund(amount, reason)
            
            return response

        except Exception as e:
            logger.error(
                "Refund processing failed",
                extra={
                    'payment_id': self.payment.id,
                    'amount': str(amount),
                    'error': str(e)
                },
                exc_info=True
            )
            raise

    def handle_webhook(self, event_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Handles webhook events from payment provider
        """
        try:
            response = self.processor.handle_callback(event_data)
            
            # Handle post-webhook business logic based on event type
            if response.get('status') == 'success':
                self._handle_successful_payment(response)
            elif response.get('status') == 'refunded':
                self._handle_successful_refund(
                    Decimal(response.get('amount', 0)),
                    response.get('reason')
                )
            
            return response

        except Exception as e:
            logger.error(
                "Webhook handling failed",
                extra={
                    'payment_id': self.payment.id,
                    'event_type': event_data.get('type'),
                    'error': str(e)
                },
                exc_info=True
            )
            raise

    def _handle_successful_payment(self, response: Dict[str, Any]):
        """
        Handles business logic after successful payment
        """
        # Update subscription
        subscription = self._update_subscription()
        
        # Update site states
        self._update_site_states(subscription)
        
        # Handle any additional vendor updates
        self._update_vendor_status(subscription)

    def _handle_successful_refund(self, amount: Decimal, reason: str = None):
        """
        Handles business logic after successful refund
        """
        # Update subscription if needed
        if amount == self.payment.amount:
            self._cancel_subscription()
        
        # Update site states
        self._handle_refund_site_states()
        
        # Update vendor status if needed
        self._update_vendor_refund_status()

    def _update_subscription(self) -> VendorSubscription:
        """
        Updates or creates subscription based on payment
        """
        subscription = VendorSubscription.objects.filter(
            vendor=self.payment.vendor,
            status='active'
        ).first()

        if subscription:
            subscription.end_date = self._calculate_subscription_end_date(subscription)
            subscription.last_payment = self.payment
            subscription.save()
        else:
            subscription = VendorSubscription.objects.create(
                vendor=self.payment.vendor,
                plan=self.payment.subscription_plan,
                start_date=timezone.now(),
                end_date=self._calculate_subscription_end_date(),
                status='active',
                last_payment=self.payment
            )

        return subscription

    def _update_site_states(self, subscription: VendorSubscription):
        """
        Updates site states after successful payment
        """
        sites = self.payment.vendor.sites.all()
        
        for site in sites:
            state_manager = SiteStateManager(site)
            state_manager.handle_payment_success(self.payment)

    def _handle_refund_site_states(self):
        """
        Updates site states after refund
        """
        sites = self.payment.vendor.sites.all()
        
        for site in sites:
            state_manager = SiteStateManager(site)
            state_manager.handle_payment_failure(self.payment)

    def _calculate_subscription_end_date(self, existing_subscription=None):
        """
        Calculates subscription end date based on plan interval
        """
        if existing_subscription and existing_subscription.end_date > timezone.now():
            start_date = existing_subscription.end_date
        else:
            start_date = timezone.now()

        interval_mapping = {
            'month': 30,
            'year': 365,
            'week': 7,
            'day': 1
        }

        days = interval_mapping.get(self.payment.subscription_plan.interval, 30)
        return start_date + timezone.timedelta(days=days)

    def _update_vendor_status(self, subscription: VendorSubscription):
        """
        Updates vendor status after successful payment
        """
        vendor = self.payment.vendor
        vendor.subscription_status = 'active'
        vendor.current_subscription = subscription
        vendor.last_payment_date = timezone.now()
        vendor.save()

    def _update_vendor_refund_status(self):
        """
        Updates vendor status after refund
        """
        vendor = self.payment.vendor
        if not vendor.has_active_subscription():
            vendor.subscription_status = 'cancelled'
            vendor.current_subscription = None
            vendor.save()

    def _cancel_subscription(self):
        """
        Cancels active subscription after full refund
        """
        subscription = self.payment.vendor.current_subscription
        if subscription:
            subscription.status = 'cancelled'
            subscription.cancelled_at = timezone.now()
            subscription.cancellation_reason = 'refund'
            subscription.save()

    @staticmethod
    def create_subscription_checkout(subscription_plan, vendor, success_url=None, cancel_url=None):
        """Create Stripe Checkout Session for subscription"""
        try:
            if not vendor.stripe_customer_id:
                customer = stripe.Customer.create(
                    email=vendor.user.email,
                    metadata={
                        'vendor_id': vendor.id,
                        'company_name': vendor.company_name
                    }
                )
                vendor.stripe_customer_id = customer.id
                vendor.save()
            
            checkout_session = stripe.checkout.Session.create(
                customer=vendor.stripe_customer_id,
                payment_method_types=['card'],
                line_items=[{
                    'price_data': {
                        'currency': 'usd',
                        'product_data': {
                            'name': subscription_plan.name,
                            'metadata': {
                                'plan_id': subscription_plan.id
                            }
                        },
                        'unit_amount': int(subscription_plan.price * 100),  # Convert to cents
                        'recurring': {
                            'interval': subscription_plan.billing_period,
                            'interval_count': 1,
                        }
                    },
                    'quantity': 1,
                }],
                mode='subscription',
                success_url=success_url or settings.STRIPE_SUCCESS_URL,
                cancel_url=cancel_url or settings.STRIPE_CANCEL_URL,
                metadata={
                    'vendor_id': vendor.id,
                    'plan_id': subscription_plan.id
                }
            )            
            return {
                'session_id': checkout_session.id,
                'url': checkout_session.url
            }
            
        except stripe.error.StripeError as e:
            raise ValueError(_("Payment processing error: {}").format(str(e)))

