# local_secrets/users/services/subscription_services.py 
import logging
from typing import Any, Dict, Optional
from local_secrets.payment.models import Payment, AutomaticBilling
from local_secrets.users.emails import VendorEmailNotifications 
from local_secrets.users.exceptions import QuotaExceeded
from local_secrets.users.models import Vendor, VendorSubscription, SubscriptionPlan, SubscriptionAuditLog
from django.db import transaction
from django.utils import timezone
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _
from ..models import (
    VendorSubscription, SubscriptionPlan, SubscriptionAuditLog,
    SubscriptionTransaction, SubscriptionUsageReport
)
from django.db.models import Sum, Count
from local_secrets.users.models import VendorSubscription, Vendor
from local_secrets.sites.models import Site

logger = logging.getLogger(__name__)

class SubscriptionService:
    def __init__(self, vendor):
        self.vendor = vendor
        self._load_active_subscription()

    def _load_active_subscription(self):
        self.active_subscription = VendorSubscription.objects.filter(
            vendor=self.vendor,
            status='active',
            start_date__lte=timezone.now(),
            end_date__gt=timezone.now()
        ).first()

    @transaction.atomic
    def create_subscription(self, plan, payment_method=None):
        """Create a new subscription for vendor"""
        start_date = timezone.now()
        
        # Calculate end date based on billing period
        if plan.billing_period == 'monthly':
            end_date = start_date + timezone.timedelta(days=30)
        elif plan.billing_period == 'yearly':
            end_date = start_date + timezone.timedelta(days=365)
        else:  # custom
            end_date = start_date + timezone.timedelta(days=plan.duration_days or 30)

        subscription = VendorSubscription.objects.create(
            vendor=self.vendor,
            plan=plan,
            start_date=start_date,
            end_date=end_date,
            status='pending',
            payment_status='pending'
        )

        # Create audit log
        SubscriptionAuditLog.objects.create(
            subscription=subscription,
            action='created',
            new_plan=plan,
            details={'payment_method': payment_method}
        )

        return subscription

    @transaction.atomic
    def process_payment(self, subscription, transaction_id, amount):
        """Process subscription payment"""
        transaction = SubscriptionTransaction.objects.create(
            subscription=subscription,
            amount=amount,
            transaction_id=transaction_id,
            status='pending'
        )

        try:
            # Process payment logic here
            payment_successful = True  # Replace with actual payment processing

            if payment_successful:
                subscription.payment_status = 'completed'
                subscription.status = 'active'
                transaction.status = 'completed'
            else:
                subscription.payment_status = 'failed'
                transaction.status = 'failed'
                
            subscription.save()
            transaction.save()

            return transaction.status == 'completed'

        except Exception as e:
            transaction.status = 'failed'
            transaction.save()
            raise ValidationError(f"Payment processing failed: {str(e)}")

    @transaction.atomic
    def update_usage(self, subscription=None):
        """Update subscription usage and create usage report"""
        subscription = subscription or self.active_subscription
        if not subscription:
            return

        # Count current usage
        sites_count = subscription.vendor.sites.filter(status='published').count()
        events_count = subscription.vendor.events.filter(status='published').count()

        # Update subscription quotas
        subscription.sites_quota_used = sites_count
        subscription.events_quota_used = events_count
        subscription.save()

        # Calculate percentages
        sites_percentage = (sites_count / subscription.plan.max_sites) * 100
        events_percentage = (events_count / subscription.plan.max_events) * 100

        # Create usage report
        SubscriptionUsageReport.objects.create(
            subscription=subscription,
            report_date=timezone.now().date(),
            sites_count=sites_count,
            sites_quota_percentage=sites_percentage,
            events_count=events_count,
            events_quota_percentage=events_percentage
        )

    @transaction.atomic
    def cancel_subscription(self, reason=None):
        """Cancel subscription"""
        if not self.active_subscription:
            raise ValidationError(_("No active subscription found"))

        subscription = self.active_subscription
        subscription.status = 'cancelled'
        subscription.cancelled_at = timezone.now()
        subscription.save()

        # Create audit log
        SubscriptionAuditLog.objects.create(
            subscription=subscription,
            action='cancelled',
            old_plan=subscription.plan,
            details={'reason': reason}
        )

    @transaction.atomic
    def upgrade_subscription(self, new_plan):
        """Upgrade subscription to new plan"""
        if not self.active_subscription:
            raise ValidationError(_("No active subscription found"))

        old_plan = self.active_subscription.plan
        
        # Validate upgrade
        if new_plan.price <= old_plan.price:
            raise ValidationError(_("New plan must be an upgrade"))

        # Update subscription
        self.active_subscription.plan = new_plan
        self.active_subscription.save()

        # Create audit log
        SubscriptionAuditLog.objects.create(
            subscription=self.active_subscription,
            action='upgraded',
            old_plan=old_plan,
            new_plan=new_plan
        )

        # Update usage limits
        self.update_usage(self.active_subscription)

class SubscriptionQuotaService:
    """
    Manages subscription quotas and usage tracking
    """
    def __init__(self, vendor: Vendor):
        self.vendor = vendor
        self._subscription = self._get_active_subscription()

    def _get_active_subscription(self) -> Optional[VendorSubscription]:
        return VendorSubscription.objects.filter(
            vendor=self.vendor,
            status='active',
            end_date__gt=timezone.now()
        ).first()

    def has_available_quota(self, site_type: str) -> bool:
        """
        Checks if vendor has available quota for given site type
        """
        if not self._subscription:
            return False

        current_usage = self._get_current_usage(site_type)
        plan_limit = self._get_plan_limit(site_type)
        
        return current_usage < plan_limit

    def check_and_increment_quota(self, site_type: str) -> bool:
        """
        Checks quota and increments if available
        Raises QuotaExceeded if quota is exceeded
        """
        if not self.has_available_quota(site_type):
            current = self._get_current_usage(site_type)
            limit = self._get_plan_limit(site_type)
            raise QuotaExceeded(_(
                f"Quota exceeded for {site_type}. Current usage: {current}, Limit: {limit}"
            ))
        
        return True

    def _get_current_usage(self, site_type: str) -> int:
        """
        Gets current usage count for site type
        """
        return Site.objects.filter(
            vendor=self.vendor,
            type=site_type,
            visibility_state='active'
        ).count()

    def _get_plan_limit(self, site_type: str) -> int:
        """
        Gets plan limit for site type
        """
        if not self._subscription:
            return 0
        
        plan_limits = self._subscription.plan.features.get('limits', {})
        return plan_limits.get(f'{site_type}_limit', 0)

    def get_quota_status(self, site_type: Optional[str] = None) -> Dict[str, Any]:
        """
        Gets detailed quota status
        """
        if not self._subscription:
            return {'status': 'no_subscription'}

        quotas = {}
        site_types = ['place', 'event'] if site_type is None else [site_type]

        for type_ in site_types:
            current = self._get_current_usage(type_)
            limit = self._get_plan_limit(type_)
            quotas[type_] = {
                'used': current,
                'limit': limit,
                'available': max(0, limit - current),
                'percentage': (current / limit * 100) if limit > 0 else 100
            }

        return quotas

class SubscriptionUsageTracker:
    def __init__(self, subscription: VendorSubscription):
        self.subscription = subscription
    
    @transaction.atomic
    def track_usage(self, resource_type: str, count: int = 1) -> None:
        """
        Tracks resource usage and enforces quota limits
        Raises QuotaExceeded if quota would be exceeded
        """
        try:
            report = self.subscription.usage_reports.select_for_update().latest('created_at')
            
            if resource_type == 'sites':
                new_count = report.sites_count + count
                if new_count > self.subscription.plan.max_sites:
                    raise QuotaExceeded(_(
                        f"Sites quota exceeded. Current: {report.sites_count}, "
                        f"Requested: {count}, Limit: {self.subscription.plan.max_sites}"
                    ))
                report.sites_count = new_count
                
            elif resource_type == 'events':
                new_count = report.events_count + count
                if new_count > self.subscription.plan.max_events:
                    raise QuotaExceeded(_(
                        f"Events quota exceeded. Current: {report.events_count}, "
                        f"Requested: {count}, Limit: {self.subscription.plan.max_events}"
                    ))
                report.events_count = new_count
            else:
                raise ValueError(f"Invalid resource type: {resource_type}")
                
            report.calculate_percentages()
            report.save()
            
            # Log usage
            logger.info(
                f"Resource usage tracked: {resource_type}",
                extra={
                    'subscription_id': self.subscription.id,
                    'resource_type': resource_type,
                    'count': count,
                    'new_total': new_count
                }
            )

        except SubscriptionUsageReport.DoesNotExist:
            # Create initial report if none exists
            self._create_initial_report()
            self.track_usage(resource_type, count)

    def _create_initial_report(self) -> SubscriptionUsageReport:
        """Creates initial usage report"""
        return SubscriptionUsageReport.objects.create(
            subscription=self.subscription,
            report_date=timezone.now().date(),
            sites_count=0,
            events_count=0,
            sites_quota_percentage=0,
            events_quota_percentage=0
        )



