
# local_secrets/users/tests/test_bulk_operations.py

from django.test import TestCase
from django.utils import timezone
from ..services.bulk_operations import BulkOperationsService
from ..models import Site, Vendor, VendorSubscription
from .factories import SiteFactory, VendorFactory, SubscriptionFactory

class BulkOperationsTests(TestCase):
    def setUp(self):
        self.vendor = VendorFactory()
        self.sites = [SiteFactory(vendor=self.vendor) for _ in range(3)]
        self.subscription = SubscriptionFactory(vendor=self.vendor)

    def test_bulk_site_operations(self):
        site_ids = [site.id for site in self.sites]
        
        # Test publish
        result = BulkOperationsService.bulk_site_operations(
            site_ids, 'publish', user=self.vendor.user
        )
        self.assertEqual(result['processed'], 3)
        self.assertEqual(result['failed'], 0)

        # Test unpublish
        result = BulkOperationsService.bulk_site_operations(
            site_ids, 'unpublish', user=self.vendor.user
        )
        self.assertEqual(result['processed'], 3)
        
        # Test invalid action
        with self.assertRaises(ValidationError):
            BulkOperationsService.bulk_site_operations(
                site_ids, 'invalid_action'
            )

    def test_bulk_vendor_operations(self):
        vendors = [VendorFactory() for _ in range(3)]
        vendor_ids = [v.id for v in vendors]
        
        result = BulkOperationsService.bulk_vendor_operations(
            vendor_ids, 'activate'
        )
        self.assertEqual(result['processed'], 3)
        
        # Verify activation
        for vendor in Vendor.objects.filter(id__in=vendor_ids):
            self.assertTrue(vendor.is_active)

    def test_bulk_subscription_updates(self):
        subscriptions = [SubscriptionFactory() for _ in range(3)]
        subscription_ids = [s.id for s in subscriptions]
        
        result = BulkOperationsService.bulk_subscription_updates(
            subscription_ids, 'extend', days=30
        )
        self.assertEqual(result['processed'], 3)
        
        # Verify extension
        for subscription in VendorSubscription.objects.filter(id__in=subscription_ids):
            self.assertTrue(
                subscription.end_date > timezone.now() + timedelta(days=29)
            )
