# local_secrets/users/tests/test_subscription_quota.py

import pytest
from datetime import timedelta
from decimal import Decimal
from django.utils import timezone
from django.test import TestCase
from django.core import mail
from unittest.mock import patch

from local_secrets.users.models import (
    Vendor, VendorSubscription, SubscriptionPlan,
    SubscriptionUsageReport, CustomUser  # Import CustomUser instead of User
)
from local_secrets.users.exceptions import QuotaExceeded
from local_secrets.users.services.subscription_services import (
    SubscriptionQuotaService,
    SubscriptionUsageTracker
)
from local_secrets.sites.models import Site

@pytest.fixture
def subscription_plan():
    return SubscriptionPlan.objects.create(
        name="Business Plan",
        price=Decimal("99.99"),
        features={
            'limits': {
                'place_limit': 10,
                'event_limit': 5,
                'image_limit': 50,
                'video_limit': 10,
                'staff_limit': 3,
                'api_calls_limit': 1000
            }
        }
    )

@pytest.fixture
def test_user():
    return CustomUser.objects.create_user(
        username="testuser",
        email="test@example.com",
        password="testpass123"
    )

@pytest.fixture
def vendor(test_user):
    return Vendor.objects.create(
        user=test_user,
        company_name="Test Company",
        business_type_id=1
    )

@pytest.fixture
def active_subscription(vendor, subscription_plan):
    subscription = VendorSubscription.objects.create(
        vendor=vendor,
        plan=subscription_plan,
        status='active',
        start_date=timezone.now(),
        end_date=timezone.now() + timedelta(days=30)
    )
    return subscription

class TestSubscriptionQuotaService(TestCase):
    def setUp(self):
        self.user = CustomUser.objects.create_user(
            username="testuser",
            email="test@example.com",
            password="testpass123"
        )
        self.vendor = Vendor.objects.create(
            user=self.user,
            company_name="Test Company",
                business_type_id=1
        )
        self.plan = SubscriptionPlan.objects.create(
            name="Business Plan",
            price=Decimal("99.99"),
            features={
                'limits': {
                    'place_limit': 10,
                    'event_limit': 5,
                    'image_limit': 50,
                    'video_limit': 10,
                    'staff_limit': 3,
                    'api_calls_limit': 1000
                }
            }
        )
        self.subscription = VendorSubscription.objects.create(
            vendor=self.vendor,
            plan=self.plan,
            status='active',
            start_date=timezone.now(),
            end_date=timezone.now() + timedelta(days=30)
        )
        self.quota_service = SubscriptionQuotaService(self.vendor)

    def test_has_available_quota_places(self):
        # Create 9 places
        for i in range(9):
            Site.objects.create(
                vendor=self.vendor,
                type='place',
                name=f"Place {i}"
            )
        
        self.assertTrue(self.quota_service.has_available_quota('place'))
        
        # Add one more place
        Site.objects.create(
            vendor=self.vendor,
            type='place',
            name="Last Place"
        )
        
        self.assertFalse(self.quota_service.has_available_quota('place'))

    def test_quota_exceeded_exception(self):
        # Fill up the quota
        for i in range(10):
            Site.objects.create(
                vendor=self.vendor,
                type='place',
                name=f"Place {i}"
            )
        
        with self.assertRaises(QuotaExceeded):
            self.quota_service.check_and_increment_quota('place')

    def test_quota_status_reporting(self):
        # Create some sites
        for i in range(5):
            Site.objects.create(
                vendor=self.vendor,
                type='place',
                name=f"Place {i}"
                
            )
        for i in range(2):
            Site.objects.create(
                vendor=self.vendor,
                type='event',
                name=f"Event {i}"
            )
        
        status = self.quota_service.get_quota_status()
        
        self.assertEqual(status['place']['used'], 5)
        self.assertEqual(status['place']['available'], 5)
        self.assertEqual(status['place']['percentage'], 50)
        self.assertEqual(status['event']['used'], 2)
        self.assertEqual(status['event']['available'], 3)
        self.assertEqual(status['event']['percentage'], 40)

class TestSubscriptionUsageTracker(TestCase):
    def setUp(self):
        self.user = CustomUser.objects.create_user(
            username="testuser",
            email="test@example.com",
            password="testpass123"
        )
        self.vendor = Vendor.objects.create(
            user=self.user,
            company_name="Test Company",
            business_type_id=1
        )
        self.plan = SubscriptionPlan.objects.create(
            name="Business Plan",
            price=Decimal("99.99"),
            features={
                'limits': {
                    'place_limit': 10,
                    'event_limit': 5
                }
            }
        )
        self.subscription = VendorSubscription.objects.create(
            vendor=self.vendor,
            plan=self.plan,
            status='active',
            start_date=timezone.now(),
            end_date=timezone.now() + timedelta(days=30)
        )
        self.tracker = SubscriptionUsageTracker(self.subscription)
        self.usage_report = SubscriptionUsageReport.objects.create(
            subscription=self.subscription,
            report_date=timezone.now().date(),
            sites_count=0,
            events_count=0
        )

    def test_track_usage_success(self):
        self.tracker.track_usage('sites', 2)
        report = self.subscription.usage_reports.latest('created_at')
        self.assertEqual(report.sites_count, 2)

    def test_track_usage_exceeds_quota(self):
        with self.assertRaises(QuotaExceeded):
            self.tracker.track_usage('sites', 11)

    @patch('local_secrets.users.services.subscription_services.logger')
    def test_usage_logging(self, mock_logger):
        self.tracker.track_usage('sites', 1)
        mock_logger.info.assert_called_with(
            "Resource usage tracked: sites",
            extra={
                'subscription_id': self.subscription.id,
                'resource_type': 'sites',
                'count': 1,
                'new_total': 1
            }
        )

class TestQuotaNotifications(TestCase):
    def setUp(self):
        self.user = CustomUser.objects.create_user(
            username="testuser",
            email="test@example.com",
            password="testpass123"
        )
        self.vendor = Vendor.objects.create(
            user=self.user,
            company_name="Test Company",
            business_type_id=1
        )
        self.plan = SubscriptionPlan.objects.create(
            name="Business Plan",
            price=Decimal("99.99"),
            features={
                'limits': {
                    'place_limit': 10,
                    'event_limit': 5
                }
            }
        )
        self.subscription = VendorSubscription.objects.create(
            vendor=self.vendor,
            plan=self.plan,
            status='active',
            start_date=timezone.now(),
            end_date=timezone.now() + timedelta(days=30)
        )
        self.tracker = SubscriptionUsageTracker(self.subscription)

    def test_quota_warning_notification(self):
        # Fill up 90% of quota
        self.tracker.track_usage('sites', 9)
        self.assertEqual(len(mail.outbox), 1)
        self.assertIn("90% of your site quota", mail.outbox[0].subject)

    def test_multiple_resource_warning(self):
        self.tracker.track_usage('sites', 9)
        self.tracker.track_usage('events', 4)
        self.assertEqual(len(mail.outbox), 2)
        self.assertTrue(
            any("site quota" in m.subject for m in mail.outbox)
        )
        self.assertTrue(
            any("event quota" in m.subject for m in mail.outbox)
        )
