# tests/test_vendor_registration.py
import pytest
from django.test import TestCase
from django.core import mail
from rest_framework.test import APITestCase
from rest_framework import status 
from ..services.email_service import VendorEmailService 
from django.contrib.auth import get_user_model  
from ..models import (
    Vendor, 
    VendorDocument, 
    CustomUser, 
    BusinessType,
    BusinessCategory
)
 
class VendorRegistrationTests(APITestCase):
    @classmethod
    def setUpTestData(cls):
        # Create a business category
        cls.business_category = BusinessCategory.objects.create(
            name='Test Category',
            slug='test-category',
            description='Test Category Description'
        )
        
        # Create a test business type with category
        cls.business_type = BusinessType.objects.create(
            category=cls.business_category,
            name='Test Business Type',
            slug='test-business-type',
            description='Test Description'
        )

    def setUp(self):
        # Create a test user
        self.user = CustomUser.objects.create_user(
            username='testuser',
            email='test@vendor.com',
            password='SecurePass123!'
        )
        self.client.force_authenticate(user=self.user)

        self.valid_payload = {
            'company_name': 'Test Company LLC',
            'contact_email': 'test@vendor.com',
            'description': 'Test company description',
            'business_type': self.business_type.id
        }

    def test_successful_registration(self):
        response = self.client.post(
            '/users/register_as_vendor',  # Removed trailing slash
            self.valid_payload,
            format='json'
        )
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertTrue(Vendor.objects.filter(
            company_name=self.valid_payload['company_name']
        ).exists())

    def test_invalid_email_registration(self):
        self.valid_payload['contact_email'] = 'invalid-email'
        response = self.client.post(
            '/users/register_as_vendor',  # Removed trailing slash
            self.valid_payload,
            format='json'
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertIn('contact_email', response.data)

    def test_welcome_email_sent(self):
        # Create a different user for this test
        test_user = CustomUser.objects.create_user(
            username='testvendor2',
            email='test2@vendor.com',
            password='TestPass123!'
        )
        
        # Create vendor with all required fields
        vendor = Vendor.objects.create(
            user=test_user,
            company_name='Test Company',
            contact_email='test2@vendor.com',
            business_type=self.business_type,
            slug='test-company'
        )
        
        # Send welcome email
        from ..emails_users import VendorEmailNotifications
        VendorEmailNotifications.send_subscription_created(vendor)
        
        self.assertEqual(len(mail.outbox), 1)
        self.assertIn(vendor.company_name, mail.outbox[0].subject)

    def tearDown(self):
        # Clean up created objects for each test
        Vendor.objects.all().delete()
        CustomUser.objects.all().delete()

    @classmethod
    def tearDownClass(cls):
        # Clean up shared test data
        BusinessType.objects.all().delete()
        BusinessCategory.objects.all().delete()
        super().tearDownClass()
