from http import HTTPStatus
import asyncio
import aiohttp
import logging

logger = logging.getLogger(__name__)

class Nominatim:
    """
    A class to interact with the Nominatim API for reverse geocoding.
    """

    base_url = "https://nominatim.openstreetmap.org/reverse"
    languages = ['es', 'en', 'fr', 'en-GB']
    address_info = {}

    def _format_address(self, response: dict, language: str):
        try:
            address_details = response["features"][0].get(
                "properties").get("address")
        except (KeyError, IndexError, AttributeError) as e:
            logger.error(f"Exception when formating address: {str(e)}")
            return None

        return {
            language: address_details
        }

    async def _fetch_location(
            self, session: aiohttp.ClientSession, latitude: float,
            longitude: float, language: str):
        """
        Retrieve the address for given latitude and longitude in multiple languages.

        Parameters:
        - session: Shared session for making requests.
        - lat: Latitude coordinate.
        - lon: Longitude coordinate.
        - language: language codes to retrieve addresses in the specified languages.

        Returns:
        - dict: A dictionary mapping language codes to addresses."
        """
        params = {
            "format": "geojson",
            "lat": str(latitude),
            "lon": str(longitude),
            "accept-language": language
        }
        headers = {"User-Agent": "MyGeocodingApp/1.0"}

        try:
            async with session.get(self.base_url, params=params, headers=headers) as response:
                if response.status == HTTPStatus.OK:
                    response_data = await response.json()
                    return self._format_address(response_data, language)
                else:
                    logger.error(f"Error {response.status} for {language}: {await response.text()}")

        except Exception as e:
            logger.error(f"Exception: {e}")

    async def get_location_data(self, latitude: int, longitude: int):
        """
        Use asyncio and aiohttp to execute API calls in parallel.
        """

        async with aiohttp.ClientSession() as session:
            tasks = {
                lang: self._fetch_location(
                    session, latitude, longitude, lang)
                for lang in self.languages}
            results = await asyncio.gather(*tasks.values())

        response_dict = {}
        for result in results:
            response_dict.update(result)

        return response_dict
