"""Digital Ocean space configurations"""
import boto3
from config.settings import base


class S3:
    """
    A utility class for managing S3 client connections to DigitalOcean Spaces.

    This class provides a way to configure and access an S3 client instance
    with credentials and endpoint configurations.

    S3 client is created and reused across the application.
    """

    # Class-level constants for S3 configuration
    _ACCESS_KEY = base.AWS_ACCESS_KEY_ID
    _SECRET_KEY = base.AWS_SECRET_ACCESS_KEY
    _REGION = base.AWS_S3_REGION_NAME
    ENDPOINT_URL = f"https://{_REGION}.digitaloceanspaces.com"

    def __init__(self):
        self._client_instance = None

    @property
    def client(self):
        """Lazy-loaded property to get or create an S3 client instance.
        """
        try:
            if self._client_instance is None:
                self._client_instance = boto3.client(
                    's3',
                    endpoint_url=self.ENDPOINT_URL,
                    aws_access_key_id=self._ACCESS_KEY,
                    aws_secret_access_key=self._SECRET_KEY,
                    region_name=self._REGION
                )
            return self._client_instance
        except Exception as e:
            raise Exception(
                f"Failed to connect to Digital ocean space: {str(e)}") from e

    def _get_bucket_region(self, aws_config: dict, bucket: str):
        bucket_client = boto3.client("s3", **aws_config)
        bucket_location = bucket_client.get_bucket_location(Bucket=bucket)
        return bucket_location["LocationConstraint"]

    def upload_to_s3(self, bucket_name, file_path, data, extension='jpeg'):
        """Upload files to digiatl ocean space.
        """
        try:
            # Get/Create the connection.
            cursor = self.client  # connect
            cursor.upload_fileobj(
                data,
                bucket_name,
                file_path,
                ExtraArgs={
                    'ACL': 'public-read',
                    'ContentType': f"image/{extension}"
                }
            )
        except Exception as e:
            raise Exception(
                f"Failed to upload to Digital ocean space: {str(e)}") from e
