
from local_secrets.cities.models import (
    City, Region, County, Country, TranslatedRegion, TranslatedCounty,
    Neighborhood, TranslatedNeighborhood)
from local_secrets.languages.models import Language


class DataMigration:
    def __init__(self, address_info: dict, city: City = None,
                 country: Country = None):
        self.address_info = dict(sorted(
            address_info.items(), key=lambda x: (x[0] != 'es', x[0]))
        )
        self.city = city
        self.country = country

    def _get_language(self, code: str):
        return Language.objects.filter(code=code).last()

    def _resolve_region(self, lang_code):
        if lang_code == 'es':
            self.region_obj, _ = Region.objects.get_or_create(
                name__iexact=self.region_name,
                defaults={
                    "name": self.region_name,
                    "country": self.country
                }
            )
        else:
            TranslatedRegion.objects.get_or_create(
                name__iexact=self.region_name,
                language=self.language_obj.id,
                region=self.region_obj.id,
                defaults={
                    "name": self.region_name,
                    "language": self.language_obj,
                    "region": self.region_obj
                }
            )

    def _resolve_county(self, lang_code):
        if lang_code == 'es':
            self.county_obj, _ = County.objects.get_or_create(
                name__iexact=self.county_name,
                region=self.region_obj.id,
                defaults={
                    "name": self.county_name,
                    "region": self.region_obj
                }
            )
        else:
            TranslatedCounty.objects.get_or_create(
                name__iexact=self.county_name,
                language=self.language_obj.id,
                county=self.county_obj.id,
                defaults={
                    "name": self.county_name,
                    "language": self.language_obj,
                    "county": self.county_obj
                }
            )

    def _resolve_neighbourhood(self, lang_code):
        if lang_code == 'es':
            self.neighbour_obj, _ = Neighborhood.objects.get_or_create(
                name__iexact=self.neighbour,
                city=self.city.id,
                defaults={
                    "name": self.neighbour,
                    "city": self.city
                }
            )
        else:
            TranslatedNeighborhood.objects.get_or_create(
                name__iexact=self.neighbour,
                language=self.language_obj.id,
                neighborhood=self.neighbour_obj.id,
                defaults={
                    "name": self.neighbour,
                    "language": self.language_obj,
                    "neighborhood": self.neighbour_obj
                }
            )

    def sync_region_and_county(self):
        for lang_code, location in self.address_info.items():
            try:
                # Retrieve language
                self.language_obj = self._get_language(lang_code)

                self.county_name = next(
                    (location.get(key)
                     for key
                     in
                     ["county", "state_district", "district", "city_district"]
                     if key in location),
                    None)

                self.region_name = next((location.get(key) for key in [
                    "state"] if key in location), None)

                if self.region_name:
                    self._resolve_region(lang_code)
        
                    if self.county_name:
                        self._resolve_county(lang_code)

            except Exception as e:
                import traceback
                print(traceback.format_exc())
                print("eroor on regionn", str(e))

    def sync_neighbour(self):
        for lang_code, location in self.address_info.items():
            try:
                # Retrieve language
                self.language_obj = self._get_language(lang_code)

                self.neighbour = next((location.get(key)
                                       for key in
                                       ["neighbourhood", "suburb"]
                                       if key in location),
                                      None)
                if self.neighbour:
                    self._resolve_neighbourhood(lang_code)

            except Exception as e:
                import traceback
                print(traceback.format_exc())
                print("eroor on neighbour", str(e))
