import decimal
from django.contrib.gis.db.models import PointField
from django.contrib.gis.geos import Point
from django.core.validators import FileExtensionValidator
from django.db import models
from django.db.models.functions import Lower
from django.utils.translation import gettext_lazy as _
from easy_thumbnails.fields import ThumbnailerImageField
from local_secrets.languages.models import Language
from django.contrib.gis.db import models as gis_models

class Continent(models.Model):
    name = models.CharField(max_length=255, verbose_name=_("Continent Name"))
    code = models.CharField(max_length=2, unique=True, verbose_name=_("ISO Code"))
    image = ThumbnailerImageField(upload_to='continent_images', verbose_name=_('Image'), null=True, blank=True)
    
    class Meta:
        verbose_name = _("Continent")
        verbose_name_plural = _("Continents")
    
    def __str__(self):
        return self.name
    
    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            return getattr(self, field)

class TranslatedContinent(models.Model):
    continent = models.ForeignKey(Continent, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=255, verbose_name=_("Translated Continent Name"))
    
    class Meta:
        unique_together = ('continent', 'language')
    
    def __str__(self):
        return f"{self.name} ({self.language})"

class Country(models.Model):
    name = models.CharField(max_length=500, verbose_name=_("Name"))
    code = models.CharField(max_length=3, verbose_name=_("ISO Code"))
    phone_code = models.CharField(max_length=10, default='34', verbose_name=_("Phone Code"))
    continent = models.ForeignKey(Continent, null=True, blank=True, on_delete=models.CASCADE, related_name="countries", verbose_name=_("Continent"))

    class Meta:
        verbose_name = _('Country')
        verbose_name_plural = _('Countries')

    def __str__(self):
        return f'{self.name} - {self.code}'

    def display_text(self, field, language='en'):
        try:
            translated_value = next(
                (t for t in self.translations.all() if t.language.code == language),
                None
            )
            return getattr(translated_value, field) if translated_value else getattr(self, field)
        except AttributeError:
            return getattr(self, field)

class TranslatedCountry(models.Model):
    country = models.ForeignKey(Country, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=500, verbose_name=_("Translated Country Name"))

    class Meta:
        unique_together = ('country', 'language')

    def __str__(self):
        return f"{self.name} ({self.language})"

class Region(models.Model):
    name = models.CharField(max_length=255, verbose_name=_("Region/State Name"))
    country = models.ForeignKey(Country, on_delete=models.CASCADE, related_name="regions")

    class Meta:
        verbose_name = _("Region")
        verbose_name_plural = _("Regions")

    def __str__(self):
        return f"{self.name}, {self.country.name}"
    
    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            return getattr(self, field)

class TranslatedRegion(models.Model):
    region = models.ForeignKey(Region, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=255, verbose_name=_("Translated Region Name"))

    class Meta:
        unique_together = ('region', 'language')

    def __str__(self):
        return f"{self.name} ({self.language})"

class County(models.Model):
    name = models.CharField(max_length=255, verbose_name=_("County/District Name"))
    region = models.ForeignKey(Region, on_delete=models.CASCADE, related_name="counties")

    class Meta:
        verbose_name = _("County")
        verbose_name_plural = _("Counties")

    def __str__(self):
        return f"{self.name}, {self.region.name}"
    
    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            return getattr(self, field)

class TranslatedCounty(models.Model):
    county = models.ForeignKey(County, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=255, verbose_name=_("Translated County Name"))

    class Meta:
        unique_together = ('county', 'language')

    def __str__(self):
        return f"{self.name} ({self.language})"

class City(models.Model):
    name = models.CharField(max_length=500, verbose_name=_("Name"))
    cp = models.CharField(max_length=12, verbose_name=_('CP'))
    province = models.CharField(max_length=100, verbose_name=_('Province'))
    point = gis_models.PointField(blank=True, null=True, verbose_name=_("Geolocation"))
    description = models.TextField(verbose_name=_('Description'))
    slogan = models.CharField(max_length=100, verbose_name=_('Slogan'))
    link = models.CharField(max_length=100, null=True, blank=True, verbose_name=_('Link'))
    media = models.FileField(
        upload_to='site_videos',
        null=True,
        blank=True,
        validators=[FileExtensionValidator(allowed_extensions=['mp4', 'avi', 'mov', 'webm', 'mkv'])],
        verbose_name=_('Video')
    ) 
    continent = models.ForeignKey('Continent', on_delete=models.SET_NULL, null=True, blank=True, related_name="cities")
    region = models.ForeignKey(Region, on_delete=models.CASCADE, related_name="cities", null=True, blank=True)
    country = models.ForeignKey(Country, on_delete=models.CASCADE, verbose_name=_('Country'))
    county = models.ForeignKey(County, on_delete=models.SET_NULL, null=True, blank=True, related_name="cities")
    latitude = models.DecimalField(max_digits=30, decimal_places=27, default=0, verbose_name=_("Latitude"))
    longitude = models.DecimalField(max_digits=30, decimal_places=27, default=0, verbose_name=_("Longitude"))
    activated = models.BooleanField(default=False)    

    class Meta:
        verbose_name = _('City')
        verbose_name_plural = _('Cities')
        ordering = [

            Lower('name'),
        ]

    def __str__(self):
        return f'{self.name}'

    def save(self, *args, **kwargs):
        if self.point:
            self.longitude = self.point.coords[0]
            self.latitude = self.point.coords[1]
        if decimal.Decimal(self.longitude).is_zero():
            self.longitude = decimal.Decimal(0.0)
        if decimal.Decimal(self.latitude).is_zero():
            self.longitude = decimal.Decimal(0.0)
        self.point = Point((decimal.Decimal(self.longitude), decimal.Decimal(self.latitude)), srid=4326)
        return super(City, self).save(*args, **kwargs)

    def display_text(self, field, language='en'):
        try:
            translated_value = next(
                (t for t in self.translations.all() if t.language.code == language),
                None
            )
            return getattr(translated_value, field) if translated_value else getattr(self, field)
        except AttributeError:
            return getattr(self, field)

class TranslatedCity(models.Model):
    city = models.ForeignKey(City, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=500, verbose_name=_("Translated City Name"))
    province = models.CharField(max_length=100, verbose_name=_('Province'))
    description = models.TextField(verbose_name=_('Description'))
    slogan = models.CharField(max_length=100, verbose_name=_('Slogan'))

    class Meta:
        unique_together = ('city', 'language')

    def __str__(self):
        return f"{self.name} ({self.language})"

class Neighborhood(models.Model):
    name = models.CharField(max_length=255, verbose_name=_("Neighborhood Name"))
    city = models.ForeignKey(City, on_delete=models.CASCADE, related_name="neighborhoods")

    class Meta:
        verbose_name = _("Neighborhood")
        verbose_name_plural = _("Neighborhoods")

    def __str__(self):
        return f"{self.name}, {self.city.name}"

class TranslatedNeighborhood(models.Model):
    neighborhood = models.ForeignKey(Neighborhood, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=255, verbose_name=_("Translated Neighborhood Name"))

    class Meta:
        unique_together = ('neighborhood', 'language')

    def __str__(self):
        return f"{self.name} ({self.language})"

class PhoneCode(models.Model):
    name = models.CharField(max_length=500, verbose_name=_('Name'))

    code = models.CharField(max_length=3, verbose_name=_('ISO Code'))
    phone_code = models.CharField(max_length=10, default=34, verbose_name=_('Phone code'))

    class Meta:
        verbose_name = _('Phone Code')
        verbose_name_plural = _('Phone Code')

    def __str__(self):
        return f'{self.name} - {self.code}'

    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            return getattr(self, field)
 
class TranslatedPhoneCode(models.Model):
    phone_code = models.ForeignKey(PhoneCode, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=500, verbose_name=_('Translated Name'))
    
class CityImage(models.Model):
    city = models.ForeignKey(City, on_delete=models.CASCADE, related_name='images', verbose_name=_('City'))
    image = ThumbnailerImageField(upload_to='city_images', verbose_name=_('Image'))

    class Meta:
        verbose_name = _('City Image')
        verbose_name_plural = _('City Images')

    def __str__(self):
        return f"Image for {self.city.name}"
 
class Address(models.Model):
    street = models.CharField(max_length=250, verbose_name=_('Street'))
    city = models.ForeignKey(City, on_delete=models.CASCADE, related_name='addresses')
    cp = models.CharField(max_length=12, verbose_name=_('CP'))  
    point = PointField(blank=True, verbose_name=_('Geolocation'))   
    latitude = models.DecimalField(max_digits=30, blank=True, decimal_places=27, default=0, verbose_name=_('Latitude'))
    longitude = models.DecimalField(
        max_digits=30, blank=True, decimal_places=27, default=0, verbose_name=_('Longitude')
    )
    neighborhood = models.ForeignKey(Neighborhood, on_delete=models.SET_NULL, null=True, blank=True, related_name="addresses")
    #country = models.ForeignKey(Country, on_delete=models.CASCADE, related_name="addresses")
 
    google_place_id = models.CharField(max_length=500, null=True, blank=True, verbose_name=_('Google Place Id'))
    details = models.CharField(max_length=500, null=True, blank=True, verbose_name=_('Details'))
    number = models.CharField(max_length=6, null=True, blank=True, verbose_name=_('Number'))
    door = models.CharField(max_length=6, null=True, blank=True, verbose_name=_('Door'))
    floor = models.CharField(max_length=100, null=True, blank=True, verbose_name=_('Floor'))
    creation_date = models.DateTimeField(auto_now_add=True, verbose_name=_('Creation date'))

    class Meta:
        verbose_name = _('Address')
        verbose_name_plural = _('Addresses')

    def __str__(self):
        return f'{self.street}, {self.city.name}'

    def save(self, *args, **kwargs):
        if self.point:
            self.longitude = self.point.coords[0]
            self.latitude = self.point.coords[1]

        if not self.longitude or decimal.Decimal(self.longitude).is_zero():
            self.longitude = decimal.Decimal(0.0)

        if not self.latitude or decimal.Decimal(self.latitude).is_zero():
            self.latitude = decimal.Decimal(0.0)

        self.point = Point((decimal.Decimal(self.longitude), decimal.Decimal(self.latitude)), srid=4326)

        # prefix = "https://www.google.com/maps/place/?q=place_id:"
        # if self.google_place_id and not self.google_place_id.startswith(prefix):
        #     self.google_place_id = prefix + self.google_place_id
            
        return  super().save(*args, **kwargs)

    def display_text(self, field, language='en'):
        field_title = f'{field}_{language}'
        try:
            return getattr(self, field_title)
        except BaseException:
            return getattr(self, field)
