# local_secrets/cities/serializers.py

from rest_framework import serializers
from local_secrets.cities.models import Address, City, Continent, Country, County, Neighborhood, Region
from local_secrets.core.serializers import ThumbnailJSONSerializer
from decimal import Decimal
from django.contrib.gis.geos import Point


class CountrySerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(CountrySerializer, self).to_representation(instance)
        request = self.context.get('request')
        language = request.headers.get('language', 'es') if request else 'es'

        # Use prefetched translations if available
        translation = getattr(instance, 'prefetched_translations', None)
        if translation:
            for field in ['name']:
                representation[field] = getattr(translation[0], field, None)

        return representation

    class Meta:
        model = Country
        fields = '__all__'


class BaseCitySerializer(serializers.ModelSerializer):
    country = CountrySerializer()

    def to_representation(self, instance):
        representation = super(
            BaseCitySerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in ['name', 'province', 'description', 'slogan']:
                representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = City
        exclude = (
            'media',
            'point',
            'province',
            'description',
            'cp',
            'latitude',
            'longitude',
        )


class CitySerializer(BaseCitySerializer):
    id = serializers.IntegerField(read_only=True)
    country = CountrySerializer(read_only=True)
    images = ThumbnailJSONSerializer(alias='', read_only=True, many=True)
    num_of_routes = serializers.SerializerMethodField()
    number_of_sites = serializers.SerializerMethodField()

    def to_representation(self, instance):
        representation = super(CitySerializer, self).to_representation(instance)
        request = self.context.get('request')
        language = request.headers.get('language', 'es') if request else 'es'

        # Use prefetched translations
        translation = getattr(instance, 'prefetched_translations', None)
        if translation:
            for field in ['name', 'province', 'description', 'slogan']:
                representation[field] = getattr(translation[0], field, None)

        return representation

    def get_num_of_routes(self, obj):
        return obj.routes.count()

    def get_number_of_sites(self, obj):
        return obj.sites.count()

    class Meta:
        model = City
        fields = '__all__'


class AddressPointSerializer(serializers.ModelSerializer):
    class Meta:
        model = Address
        fields = ('street', 'latitude', 'longitude')


class AddressSerializer(serializers.ModelSerializer):
    city = serializers.PrimaryKeyRelatedField(queryset=City.objects.all())

    def to_representation(self, instance):
        representation = super(
            AddressSerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in ['details', 'street']:
                representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = Address
        fields = [
            'street', 'number', 'city', 'cp', 'latitude',
            'longitude', 'point', 'details', 'door',
            'floor', 'google_place_id'
        ]


class AddressRetrieveSerializer(serializers.ModelSerializer):
    city = CitySerializer()

    class Meta:
        model = Address
        fields = [
            'street', 'number', 'city', 'cp', 'latitude',
            'longitude', 'point', 'details', 'door',
            'floor', 'google_place_id'
        ]


class AddressCreateSerializer(serializers.ModelSerializer):
    class Meta:
        model = Address
        fields = (
            'street',
            'city',
            'cp',
            'latitude',
            'longitude',
            'details',
            'number',
            'door',
            'floor',
            'google_place_id'
        )

    def create(self, validated_data):
        # Get or calculate latitude/longitude if not provided
        latitude = validated_data.get('latitude', Decimal('0.0'))
        longitude = validated_data.get('longitude', Decimal('0.0'))

        # Create the point
        point = Point(float(longitude), float(latitude), srid=4326)
        validated_data['point'] = point

        return Address.objects.create(**validated_data)


class ContinentSerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(ContinentSerializer,
                               self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:

                for field in self.fields:
                    if field in ['name']:
                        representation[field] = instance.display_text(field, language)
                return representation
            
            except:
                pass

        return representation

    class Meta:
        model = Continent
        fields = '__all__'


class RegionSerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(
            RegionSerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:
                translation = instance.translations.get(language=language)
                representation['name'] = translation.name
            except:
                pass

        return representation

    class Meta:
        model = Region
        fields = '__all__'


class CountySerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(
            CountySerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:
                translation = instance.translations.get(language=language)
                representation['name'] = translation.name
            except:
                pass

        return representation

    class Meta:
        model = County
        fields = '__all__'


class NeighborhoodSerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(NeighborhoodSerializer,
                               self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:
                translation = instance.translations.get(language=language)
                representation['name'] = translation.name
            except:
                pass

        return representation

    class Meta:
        model = Neighborhood
        fields = '__all__'


class FullAddressSerializer(serializers.ModelSerializer):
    city = BaseCitySerializer(read_only=True)
    neighborhood = NeighborhoodSerializer(read_only=True)
    country = CountrySerializer(read_only=True)
    region = serializers.SerializerMethodField()
    county = serializers.SerializerMethodField()
    continent = serializers.SerializerMethodField()

    def get_region(self, obj):
        if obj.city and obj.city.region:
            return RegionSerializer(obj.city.region, context=self.context).data
        return None

    def get_county(self, obj):
        if obj.city and obj.city.county:
            return CountySerializer(obj.city.county, context=self.context).data
        return None

    def get_continent(self, obj):
        if obj.city and obj.city.continent:
            return ContinentSerializer(
                obj.city.continent, context=self.context).data
        return None

    def to_representation(self, instance):
        representation = super(FullAddressSerializer,
                               self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:
                translation = instance.translations.get(language=language)
                representation['street_name'] = translation.street_name
                representation['house_number'] = translation.house_number
            except:
                pass

        return representation

    class Meta:
        model = Address
        fields = '__all__'


class AddressCreateSerializer(serializers.ModelSerializer):
    city = serializers.PrimaryKeyRelatedField(
        queryset=City.objects.all(),
        required=True,
        error_messages={
            'required': 'City is required.',
            'does_not_exist': 'Selected city does not exist.',
            'incorrect_type': 'Invalid city selection.'
        }
    )

    class Meta:
        model = Address
        fields = (
            'street',
            'number',
            'city',
            'cp',
            'latitude',
            'longitude',
            'details',
            'door',
            'floor',
            'google_place_id'
        )
        extra_kwargs = {
            'door': {'required': False},
            'floor': {'required': False},
            'google_place_id': {'required': False},
            'details': {'required': False}
        }

    def create(self, validated_data):
        latitude = validated_data.get('latitude', Decimal('0.0'))
        longitude = validated_data.get('longitude', Decimal('0.0'))
        point = Point(float(longitude), float(latitude), srid=4326)
        validated_data['point'] = point
        return Address.objects.create(**validated_data)


class AddressRetrieveSerializer(serializers.ModelSerializer):
    city = CitySerializer(read_only=True)

    class Meta:
        model = Address
        fields = (
            'street', 'number', 'city', 'cp', 'latitude',
            'longitude', 'point', 'details', 'door',
            'floor', 'google_place_id'
        )

    def to_representation(self, instance):
        representation = super(AddressRetrieveSerializer,
                               self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
            for field in self.fields:
                if field in ['details', 'street']:
                    representation[field] = instance.display_text(
                        field, language)
        return representation
