# local_secrets/core/events.py

import logging
from typing import Dict, Any
from django.conf import settings
from django.utils import timezone
from django.db import models

logger = logging.getLogger(__name__)

class SystemEvent(models.Model):
    """
    Model to store system events
    """
    event_type = models.CharField(max_length=100)
    payload = models.JSONField()
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        indexes = [
            models.Index(fields=['event_type']),
            models.Index(fields=['created_at'])
        ]

class EventManager:
    """
    Handles system-wide event emission and tracking
    """
    def __init__(self):
        self.logger = logging.getLogger(__name__)

    def emit(self, event_type: str, payload: Dict[str, Any]) -> None:
        """
        Emit and store a system event
        """
        try:
            # Add timestamp to payload
            payload['timestamp'] = timezone.now().isoformat()
            
            # Store event in database
            SystemEvent.objects.create(
                event_type=event_type,
                payload=payload
            )
            
            # Log event
            self.logger.info(
                f"Event emitted: {event_type}",
                extra={
                    'event_type': event_type,
                    'payload': payload
                }
            )

        except Exception as e:
            self.logger.error(
                f"Failed to emit event {event_type}: {str(e)}",
                extra={
                    'event_type': event_type,
                    'payload': payload
                }
            )
