import traceback
import logging
from django.conf import settings
from django.utils import translation
from django.utils.translation import gettext_lazy as _
from rest_framework import status
from rest_framework.exceptions import (
    APIException,
    AuthenticationFailed,
    NotAuthenticated,
    PermissionDenied,
    ValidationError,
)
from rest_framework.response import Response
from rest_framework.views import exception_handler as default_exception_handler
from local_secrets.core.custom_exceptions import CustomApiException

logger = logging.getLogger(__name__)


def exception_handler(exception, context):
    response_status = status.HTTP_400_BAD_REQUEST
    language = translation.get_language()
    
    with translation.override(language):
        if isinstance(exception, APIException):
            payload = {
                'error': 'An error ocurred',
            }

            if (
                isinstance(exception, PermissionDenied)
                or isinstance(exception, NotAuthenticated)
                or isinstance(exception, AuthenticationFailed)
            ):
                payload['error'] = 'invalid_grant'
                response_status = status.HTTP_401_UNAUTHORIZED
                if payload.get('error_description'):
                    payload['detail'] = payload.pop('error_description')

            if type(exception) == CustomApiException:
                payload['detail'] = exception.detail
                response_status = exception.status_code
            
            elif type(exception) == ValidationError:
                try:
                    payload['code'] = exception.detail.get('code')
                except AttributeError:
                    payload['code'] = None
                
                if type(exception.detail) == list:
                    payload['detail'] = ", ".join(exception.detail)
                else:
                    fields = list(exception.detail.keys())
                    if 'detail' in fields and 'code' in fields:
                        payload['detail'] = exception.detail.get('detail')
                    else:
                        error_fields = ", ".join(fields)
                        payload['detail'] = _('The fields ') + error_fields + _(' are required')        
            else:
                if settings.DEBUG or settings.ENVIRONMENT == 'test':
                    payload['detail'] = exception.detail if isinstance(exception.detail, dict) else str(exception)
        else:
            return default_exception_handler(exception, context)
        
        return Response(data=payload, status=response_status)

def log_exception(exception: Exception):
    """
    Logs detailed information about an exception, including the traceback, filename, line number, 
    the code that caused the error, and the exception message.
    """
    if tb := traceback.extract_tb(exception.__traceback__):
        last_traceback = tb[-1]
        debugger = (
            f"Error in file: {last_traceback.filename}, "
            f"Line number: {last_traceback.lineno}, "
            f"Cause: {last_traceback.line}, "
            f"Message: {str(exception)}"
        )
        logger.error("Traceback Error: %s", debugger)
