# local_secrets/users/management/commands/check_expiring_subscriptions.py

from datetime import timedelta
from django.core.management.base import BaseCommand
from django.utils import timezone
from django.conf import settings
from local_secrets.users.models import VendorSubscription
from local_secrets.users.emails import VendorEmailNotifications

class Command(BaseCommand):
    help = 'Check for subscriptions that are about to expire and send notifications'

    def add_arguments(self, parser):
        parser.add_argument(
            '--days',
            type=int,
            default=7,
            help='Number of days before expiration to send notification'
        )

    def handle(self, *args, **options):
        days_threshold = options['days']
        expiration_date = timezone.now().date() + timedelta(days=days_threshold)
        
        # Get subscriptions about to expire
        expiring_subscriptions = VendorSubscription.objects.filter(
            status='active',
            end_date__date=expiration_date,
            auto_renew=False
        ).select_related('vendor', 'vendor__user', 'plan')
        
        for subscription in expiring_subscriptions:
            try:
                VendorEmailNotifications.send_subscription_expiring_soon(subscription)
                self.stdout.write(
                    self.style.SUCCESS(
                        f'Sent expiration notice to {subscription.vendor.company_name}'
                    )
                )
            except Exception as e:
                self.stdout.write(
                    self.style.ERROR(
                        f'Failed to send notice to {subscription.vendor.company_name}: {str(e)}'
                    )
                )
