
# local_secrets/sites/management/commands/fix_site_slugs.py
from django.core.management.base import BaseCommand
from django.utils.text import slugify
from local_secrets.sites.models import Site
import random
import string


class Command(BaseCommand):
    help = 'Fixes site slugs by ensuring all sites have unique slugs'

    def handle(self, *args, **kwargs):
        self.stdout.write('Starting site slug fix...')
        
        # Get sites with no slugs
        empty_slug_sites = Site.objects.filter(slug__isnull=True).union(Site.objects.filter(slug=''))
        self.stdout.write(f'Found {empty_slug_sites.count()} sites with empty slugs')
        
        # Generate slugs for sites with empty slugs
        for site in empty_slug_sites:
            self.fix_site_slug(site)
        
        # Check for duplicate slugs
        self.check_duplicates()
        
        self.stdout.write(self.style.SUCCESS('Successfully fixed site slugs'))

    def fix_site_slug(self, site):
        original_slug = slugify(site.title or ''.join(random.choices(string.ascii_lowercase, k=8)))
        slug = original_slug
        counter = 1
        
        while Site.objects.filter(slug=slug).exclude(pk=site.pk).exists():
            slug = f"{original_slug}-{counter}"
            counter += 1
        
        site.slug = slug
        site.save(update_fields=['slug'])
        self.stdout.write(f'Set slug "{slug}" for site #{site.pk} ({site.title})')

    def check_duplicates(self):
        from django.db.models import Count
        
        duplicates = Site.objects.values('slug').annotate(count=Count('id')).filter(count__gt=1)
        if duplicates.exists():
            self.stdout.write(self.style.WARNING(f'Found {duplicates.count()} duplicate slugs!'))
            
            for dup in duplicates:
                self.stdout.write(f'Fixing duplicate slug: {dup["slug"]}')
                sites = Site.objects.filter(slug=dup['slug']).order_by('id')
                
                # Keep the first one as is, change the rest
                for site in sites[1:]:
                    self.fix_site_slug(site)
        else:
            self.stdout.write(self.style.SUCCESS('No duplicate slugs found'))

# python manage.py fix_site_slugs