# local_secrets/core/management/commands/translate_messages.py
 
import asyncio
from typing import List, Any
from django.core.management.base import BaseCommand, CommandError
from django.conf import settings
from pathlib import Path
from argparse import ArgumentParser

from local_secrets.scripts.translation_utils import TranslationConfig, TranslationManager
 
class Command(BaseCommand):
    help = 'Translate Django PO files using OpenAI API'

    def add_arguments(self, parser: ArgumentParser) -> None:
        parser.add_argument(
            '--languages',
            nargs='+',
            type=str,
            help='Target language codes (e.g., es fr de)'
        )
        parser.add_argument(
            '--update',
            action='store_true',
            help='Only update missing translations'
        )
        parser.add_argument(
            '--model',
            type=str,
            choices=['gpt-4', 'gpt-3.5-turbo'],
            default='gpt-3.5-turbo',
            help='OpenAI model to use for translation'
        )

    def validate_languages(self, langs: List[str]) -> None:
        """Validate language codes against supported languages"""
        supported_languages = {
            'es': 'Spanish',
            'fr': 'French',
            'de': 'German',
            'pt': 'Portuguese',
            'po': 'Portuguese',   
            'en': 'English',
        }
        
        for lang in langs:
            if lang not in supported_languages:
                raise CommandError(
                    f"Unsupported language code: {lang}. "
                    f"Supported languages are: {', '.join(supported_languages.keys())}"
                )
 
    async def handle_async(self, *args: Any, **options: Any) -> None:
        languages = options.get('languages', [])
        if not languages:
            raise CommandError("Please specify target languages using --languages")


        self.validate_languages(languages)  
        api_key = getattr(settings, 'OPENAI_API_KEY', None)
        if not api_key:
            raise CommandError(
                "OpenAI API key not found. Please set OPENAI_API_KEY in your Django settings."
            )


        config = TranslationConfig(
            api_key=api_key,
            base_locale_path=Path(settings.BASE_DIR) / 'locale',
            batch_size=10,
            model=options['model']  # Now this will work
        )

        self.validate_languages(languages) 
        api_key = ''
        #getattr(settings, 'OPENAI_API_KEY', None)
        if not api_key:
            raise CommandError(
                "OpenAI API key not found. Please set OPENAI_API_KEY in your Django settings."
            )

        config = TranslationConfig(
            api_key=api_key,
            base_locale_path=Path(settings.BASE_DIR) / 'locale',
            batch_size=10,  
            model=options['model']
        )
        try:
            manager = TranslationManager(config)
            
            if options['update']:
                self.stdout.write("Updating missing translations...")
                await manager.update_existing_translations(languages)
            else:
                self.stdout.write("Starting translation process...")
                for lang in languages:
                    try:
                        self.stdout.write(f"Translating to {lang}...")
                        await manager.translate_po_file(source_lang='en', target_lang=lang)
                        self.stdout.write(
                            self.style.SUCCESS(f"Successfully translated to {lang}")
                        )
                    except Exception as e:
                        self.stdout.write(
                            self.style.ERROR(f"Error translating to {lang}: {str(e)}")
                        )
            self.stdout.write(self.style.SUCCESS("Translation process completed"))
        except Exception as e:
            raise CommandError(f"Translation failed: {str(e)}")
 
    def handle(self, *args: Any, **options: Any) -> None:
        """
        Django command handle method that runs the async handle_async method
        """
        asyncio.run(self.handle_async(*args, **options))

 
        
"""
python manage.py translate_messages --languages po fr es
python manage.py translate_messages --languages po --update
python manage.py translate_messages --languages po fr --model gpt-4

"""
# python manage.py translate_messages --target-langs es
# python manage.py translate_messages --target-langs es fr de

# python manage.py translate_messages --target-langs es fr --source-lang en --batch-size 15
# python manage.py translate_messages --target-langs es fr --update-only
 
