from django.core.exceptions import ValidationError
from local_secrets.sites.models import Site
from local_secrets.travels.models import Stop

def handle_stops(travel, stops_data):
    added_stops = []

    for stop in stops_data:
        if isinstance(stop, int):
            # Try to load as a premade Stop ID
            stop_instance = Stop.objects.filter(id=stop).first()

            if stop_instance:
                # Premade itinerary stop
                added_stops.append(stop_instance)
            else:
                # If no Stop exists, interpret as a Site ID
                site_instance = Site.objects.filter(id=stop).first()
                if not site_instance:
                    raise ValidationError(f"Site with ID {stop} does not exist.")
                # Dynamically create a Stop for this Site
                stop_instance, _ = Stop.objects.get_or_create(
                    travel=travel,
                    site=site_instance,
                    defaults={"order": 0},  # Default order if not provided
                )
                added_stops.append(stop_instance)

        elif isinstance(stop, dict) and "id" in stop:
            # Handle dictionary input as user-provided Site
            site_id = stop["id"]
            site_instance = Site.objects.filter(id=site_id).first()
            if not site_instance:
                raise ValidationError(f"Site with ID {site_id} does not exist.")
            stop_instance, _ = Stop.objects.get_or_create(
                travel=travel,
                site=site_instance,
                defaults={"order": stop.get("order", 0)},  # Use provided order or 0
            )
            added_stops.append(stop_instance)

        else:
            raise ValidationError(f"Invalid stop format: {stop}")

    # Dynamically add stops to the travel
    travel.stops.add(*[s.site.id for s in added_stops])
    return added_stops