from dal import autocomplete
from django import forms
from django.contrib.admin.widgets import AutocompleteSelectMultiple
from django.utils.translation import gettext_lazy as _

from .models import Event, TranslatedEvent
from ..core.admin import admin_site
from ..sites.admin import SiteAdmin, TranslatedSiteAdmin, LevelAdmin, LevelFilter, CityFilter, CategoryFilter, \
    SubCategoryFilter
from ..sites.choices import SiteType
from ..sites.models import Level, Category, SubCategory
from ..sites.validators import video_size


class CustomAutocomplete(AutocompleteSelectMultiple):
    def get_context(self, name, value, attrs):
        context = super(CustomAutocomplete, self).get_context(name, value, attrs)
        context['is_event'] = True
        return context


class EventForm(forms.ModelForm):
    media = forms.FileField(required=False, validators=[video_size])
    type = forms.ChoiceField(choices=SiteType.choices, initial=SiteType.EVENT.value)
    levels = forms.ModelMultipleChoiceField(
        queryset=Level.objects.filter(type='event'),
        widget=autocomplete.ModelSelect2Multiple(),
    )
    categories = forms.ModelMultipleChoiceField(
        queryset=Category.objects_for_admin.filter(type='event'),
        required=True,
        widget=autocomplete.ModelSelect2Multiple(
            url='category-autocomplete',
            forward=('levels',)
        )
    )
    subcategories = forms.ModelMultipleChoiceField(
        queryset=SubCategory.objects_for_admin.filter(type='event'),
        required=False,
        widget=autocomplete.ModelSelect2Multiple(
            url='subcategory-autocomplete',
            forward=('categories',)
        )
    )

    class Meta:
        model = Event
        fields = '__all__'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # Always set type to event
        self.initial['type'] = SiteType.EVENT.value
        
        # Ensure proper querysets for related fields
        self.fields['levels'].queryset = Level.objects_for_admin.filter(type='event')
        self.fields['categories'].queryset = Category.objects_for_admin.filter(type='event')
        self.fields['subcategories'].queryset = SubCategory.objects_for_admin.filter(type='event')

    def clean(self):
        cleaned_data = super().clean()
        # Always ensure type is event
        cleaned_data['type'] = SiteType.EVENT.value
        
        categories = cleaned_data.get('categories')
        subcategories = cleaned_data.get('subcategories')

        for level in cleaned_data.get('levels', []):
            if categories and level.id not in categories.values_list('level', flat=True):
                raise forms.ValidationError({'categories': _('The selected categories must be part of the level')})
            
            if subcategories:
                for subcategory in subcategories:
                    if subcategory.category not in categories:
                        raise forms.ValidationError(
                            {'subcategories': _('The selected subcategories must be part of a selected category')}
                        )

        return cleaned_data


class EventLevelFilter(LevelFilter):
    autocomplete_url = 'event-level-autocomplete'

    def get_queryset_for_field(self, model, name):
        return Level.objects_for_admin.filter(type='event')


class EventAdmin(SiteAdmin):
    form = EventForm
    
    fieldsets = (
        (_('Basic Information'), {
            'fields': (
                'title', 
                'description', 
                'type', 
                'frequency'
            )
        }),
        (_('Location'), {
            'fields': (
                'address', 
                'city', 
                'phone', 
                'url'
            )
        }),
        (_('Relationships'), {
            'fields': (
                'levels', 
                'categories', 
                'subcategories', 
                'tags'
            )
        }),
        (_('Settings'), {
            'fields': (
                'status',
                'always_open',
                'vendor_notes',
                'vendor_contact_info'
            )
        }),
        (_('Media'), {
            'fields': ('media',)
        })
    )

    def get_queryset(self, request):
        return super().get_queryset(request).filter(type=SiteType.EVENT.value)

    def save_model(self, request, obj, form, change):
        """Ensure type is always set to event"""
        obj.type = SiteType.EVENT.value
        super().save_model(request, obj, form, change)
 
    list_filter = (
        'status',
        'frequency',
        EventLevelFilter,
        CityFilter,
        CategoryFilter,
        SubCategoryFilter
    )

    search_fields = (
        'title',
        'description',
        'address__street',
        'city__name',
        'vendor_notes',
        'vendor_contact_info'
    )
 

class TranslatedEventAdmin(TranslatedSiteAdmin):
    def get_queryset(self, request):
        return super(TranslatedEventAdmin, self).get_queryset(request).filter(site__type='event')


admin_site.register(Event, EventAdmin)
admin_site.register(TranslatedEvent, TranslatedEventAdmin)
